/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999,2000 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

#include "compiler.h"

#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
#  include <signal.h>
#  if defined(linux) && defined(HAVE_UCONTEXT_H)
#    if (__GLIBC__ == 2) && (__GLIBC_MINOR__ == 0)
typedef struct sigaltstack stack_t;
	// glibc 2.0 needs stack_t type. but cannot include <asm/signal.h>
#    endif	// glibc 2.0
#    include <asm/ucontext.h>	// neither <ucontext.h> nor <sys/ucontext.h>
#  endif
#endif	// EXC_BY_SIGNAL || GET_SIGCONTEXT


//
// Local Functions
//
static void showFPUEnv(FILE *, char *indent, char fpuenv[28]);
#define fillFPUEnv(FPUENV)	asm("fstenv %0" : "=m" (FPUENV))


//
// Definitions
//
// return from signal handler
// JDK 1.2: set program counter to jump to exception handler
// JDK 1.1: jump to exception handler */
#if JDK_VER >= 12
#  define SIGRETURN(SC, EBP)	return TRUE;
#else
#  if defined(__FreeBSD__)
#    define SIGRETURN(SC, EBP)	sigreturn(SC);
#  elif defined(sun)
#    define SIGRETURN(SC, EBP)	sigcontext(SC);
#  else	// linux
#    define SIGRETURN(SC, EBP) \
  {\
    register int eax asm("eax");\
    asm("movl  %1,%%ebp" : "=a" (eax) : "m" (EBP));\
    return eax;\
  }
#  endif	// os
#endif	// JDK_VER



//
// Variables
//
#if (defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)) && defined(SEARCH_SIGCONTEXT)
int sc_nest = -1;	// -1: invalid
static int ucontext_used = 0;
#endif


#if (defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)) && defined(SEARCH_SIGCONTEXT)
#ifdef __FreeBSD__
bool_t examineSigcontextNestCount(int sig, int code, struct sigcontext *uc0) {
#else
bool_t examineSigcontextNestCount(int sig, void *info, void *uc0) {
#endif
  SIGCONTEXT *sc, *found_sc;
  uint32_t *ebp, *found_ebp = 0;
  int i;

#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
  printf("examineSigcontextNestCount: sig %d (sys_thread_t: %x)\n",
	sig, (int)sysThreadSelf());
#endif

  asm("movl  %%ebp,%0" : "=r" (ebp));

#define SC_SEARCH_WIDTH	10
  for (i = 0; i < SC_SEARCH_WIDTH; i++) {
    extern void exam_point();	// in compiler.c

    sc = (SIGCONTEXT *)(ebp + 3);
#ifdef RUNTIME_DEBUG
    printf("  i:%d, eip:%x\n", i, sc->SC_EIP);
    fflush(stdout);
#endif
#define INT_INSN_SIZE	2	// size of int $0x10
    if ((sc->SC_EIP == ((uint32_t)exam_point))) {	// found
      sc_nest = i;
      ucontext_used = 0;	// false
      found_sc = sc;
      found_ebp = ebp;
#ifdef RUNTIME_DEBUG
      printf("  sc_nest: %d, ucontext is not used.\n", sc_nest);
      fflush(stdout);
#endif
      // break;
		// must not do break to use last found sigcontext
		// with LinuxThreads in glibc 2.1.2
    }
#ifdef HAVE_UCONTEXT_H
    else {	// not found
      struct ucontext *uc;
      uc = (struct ucontext *)ebp[4];
      if ((ebp < (uint32_t *)uc) && ((uint32_t *)uc < ebp + 0x100)) {
	sc = (SIGCONTEXT *)&uc->uc_mcontext;
	if (sc->SC_EIP == ((uint32_t)exam_point)) {	// found
	  sc_nest = i;
	  ucontext_used = 1;	// true
	  found_sc = sc;
	  found_ebp = ebp;
#ifdef RUNTIME_DEBUG
	  printf("  sc_nest: %d, ucontext is used.\n", sc_nest);
	  fflush(stdout);
#endif
	  break;
	}
      }
    }
#endif	// HAVE_UCONTEXT_H

    ebp = (uint32_t *)ebp[0];
  }

  if (sc_nest < 0) {
    printf("FATAL: cannot examine the offset of signal context.\n");
    JVM_Exit(1);
  }

#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
  printf("sigcontext: 0x%08x\n", (int)found_sc);  fflush(stdout);
  showSigcontext(found_sc);
#endif

  found_sc->SC_EIP += INT_INSN_SIZE;
  SIGRETURN(found_sc, found_ebp);
}
#endif	// (EXC_BY_SIGNAL || GET_SIGCONTEXT) && SEARCH_SIGCONTEXT


#ifdef __FreeBSD__
bool_t signalHandler(int sig, int code, struct sigcontext *uc0) {
#else
bool_t signalHandler(int sig, void *info, void *uc0) {
#endif
#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
#  ifdef SEARCH_SIGCONTEXT
  uint32_t *ebp;
#  endif	// SEARCH_SIGCONTEXT
  ExecEnv *ee = EE();
  SIGCONTEXT *sc = NULL;
  struct methodblock *mb;
  CodeInfo *codeinfo;
  uint32_t native_off;
#endif	// EXC_BY_SIGNAL || GET_SIGCONTEXT
#ifdef EXC_BY_SIGNAL
  throwentry *tentry;
  uint32_t jump_target = 0;
#endif	// EXC_BY_SIGNAL
  int i;

#if 0
#if 1
  signal(sig, SIG_DFL);
#else
  {
    struct sigaction sigAct;
    sigAct.sa_handler = SIG_DFL;
    sigemptyset(&sigAct.sa_mask);
    sigaction(sig, &sigAct, (struct sigaction *)NULL);
  }
#endif
#endif	// 0

#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
  printf("Signal handler: sig %d (sys_thread_t: %x)\n",
	sig, (int)sysThreadSelf());
#endif


#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
#ifndef SEARCH_SIGCONTEXT
#  ifdef __FreeBSD__
  sc = (SIGCONTEXT *)uc0;
#  else	// linux
  sc = (SIGCONTEXT *) &((struct ucontext *)uc0)->uc_mcontext;
#  endif
#else
  // get signal context
  asm("movl  %%ebp,%0" : "=r" (ebp));
  for (i = 0; i < sc_nest; i++) {
    ebp = (uint32_t *)ebp[0];
  }
  if (!ucontext_used) {
    sc = (SIGCONTEXT *)(ebp + 3);
  }
#ifdef HAVE_UCONTEXT_H
  else {
    sc = (SIGCONTEXT *) &((struct ucontext *)ebp[4])->uc_mcontext;
  }
#endif	// HAVE_UCONTEXT_H
#endif	// SEARCH_SIGCONTEXT

#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG) || (defined(GET_SIGCONTEXT) && !defined(EXC_BY_SIGNAL))
  printf("sigcontext: 0x%08x\n", (int)sc);  fflush(stdout);
  showSigcontext(sc);
#endif
#endif	// EXC_BY_SIGNAL || GET_SIGCONTEXT


#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
  if ((sig != SIGSEGV) && (sig != SIGFPE))  return FALSE;

  mb = ee->current_frame->current_method;
  if (!mb)  return FALSE;
#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
  printf("method(0x%x)", (int)mb);  fflush(stdout);
  printf(": %s#%s %s\n",
	cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
  fflush(stdout);
#endif
  if (!mb->CompiledCode)  return FALSE;		// not compiled
  native_off =
	((uint32_t)sc->SC_EIP) - ((uint32_t)mb->CompiledCode);
#ifdef RUNTIME_DEBUG
  printf("native off: 0x%x\n", native_off);
  fflush(stdout);
#endif
#ifndef EXC_BY_SIGNAL
  return FALSE;
#else
  codeinfo = (CodeInfo *)(mb->CompiledCodeInfo);
  tentry = throwtableGet(codeinfo, native_off);

#define IP_SEARCH_COUNT	3
  if (!tentry) {	// search EIP in native code
    uint32_t *ebp;
#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
    printf("throwentry is null !\n");  fflush(stdout);
#endif	// RUNTIME_DEBUG || COMPILE_DEBUG

    ebp = (uint32_t *)sc->SC_EBP;

    for (i = 0; i < IP_SEARCH_COUNT; i++) {
      uint32_t *sp = ebp + 1;

      native_off = *sp - (uint32_t)mb->CompiledCode;
      if ((native_off >= 0) && (native_off <= codeinfo->code_size)) {
	tentry = throwtableGet(codeinfo, native_off);
	if (tentry) {
#ifdef ARITHEXC_BY_SIGNAL
	  if (sig == SIGFPE) {
	    int opc = tentry->opcode;
	    if ((opc != opc_idiv) && (opc != opc_irem) &&
		(opc != opc_ldiv) && (opc != opc_lrem))
	      continue;
	  }
#endif	// ARITHEXC_BY_SIGNAL
	  sc->SC_ESP = (uint32_t)(sp + 1);	// stack top in the frame
	  sc->SC_EIP = *sp;			// %eip on the compiled code
	  sc->SC_EBP = ebp[0];		// return one more frame
	  goto tentry_is_found;
	}
      }
      ebp = (uint32_t *)ebp[0];
    }	// for (i = 0; i < IP_SEARCH_COUNT; ...
    goto signal_handler_error;
  tentry_is_found:
  }
#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
  printf("throwentry: ");
  fflush(stdout);
  if (tentry)
    printf("start: 0x%x, len: 0x%x, byteoff: 0x%x\n",
		tentry->start, tentry->len, tentry->byteoff);
  else
    printf("(null)\n");
  fflush(stdout);
#endif	// RUNTIME_DEBUG || COMPILE_DEBUG

#ifdef NULLEXC_BY_SIGNAL
  if (sig == SIGSEGV) {
    if (!exceptionOccurred(ee)) {
      // NullPointerException occurred
      SignalError(NULL, JAVAPKG "NullPointerException", 0);
	// set ee->exception.exc, exception object
    }
  }
#endif
#if defined(NULLEXC_BY_SIGNAL) && defined(ARITHEXC_BY_SIGNAL)
  else
#endif
#ifdef ARITHEXC_BY_SIGNAL
  if (sig == SIGFPE) {
    if ((tentry->opcode == opc_idiv) || (tentry->opcode == opc_irem)) {
      if (sc->SC_ECX /* divisor */ == 0) {
	// ArithmeticException occurred
	SignalError(NULL, JAVAPKG "ArithmeticException", "/ by zero");
      }
      else if ((sc->SC_EAX /* dividend */ == 0x80000000) &&
	       (sc->SC_ECX /* divisor  */ == -1)) {
	// idiv insn. of x86 causes it: 0x80000000 / -1
	sc->SC_EAX = 0x80000000;
	sc->SC_EDX = 0;
	jump_target = sc->SC_EIP + 2;	// `+ 2' skips idiv `0xf7 0xXX'
      }
      else {
	float dividend, divisor;
	memcpy(&dividend, &(sc->SC_EAX), 4);
	memcpy(&divisor, &(sc->SC_ECX), 4);
	printf("FATAL: SIGFPE occurred in %s: %e %s %e.\n",
		((tentry->opcode == opc_idiv)?"idiv":"irem"),
		(double)dividend, ((tentry->opcode == opc_idiv)?"/":"%"),
		(double)divisor);
	fflush(stdout);

	goto signal_handler_error;
      }
    }
    else if ((tentry->opcode == opc_ldiv) || (tentry->opcode == opc_lrem)) {
      uint32_t *sp = (uint32_t *)sc->SC_ESP;

      // `back up'ed registers in ldiv
      sc->SC_EBP = *(sp + 4);	// *(sp + 4) is %ebp
      sc->SC_EBX = *(sp + 5);	// *(sp + 5) is %ebx
      sc->SC_ESI = *(sp + 6);	// *(sp + 6) is %esi

      sc->SC_ESP += 28;	// clear arguments of __divdi3, __moddi3

      // dividend: (*(sp + 1) << 32) | *sp
      // divisor : (*(sp + 3) << 32) | *(sp + 2)

      if (!((*(sp + 2)) | (*(sp + 3)))) {
	// ArithmeticException occurred
	SignalError(NULL, JAVAPKG "ArithmeticException", "/ by zero");
      }
#if 0
      // In this case,
      // functions called by ldiv (__divdi3(),__moddi3()) don't cause SIGFPE
      else if ((*sp == 0) && (*(sp + 1) == 0x80000000) &&
	       ((*(sp +2) & *(sp + 3)) == 0xffffffff)) {
      }
#endif
    }
    else {
      printf("FATAL: SIGFPE occured in %s.",
	((tentry->opcode == opc_ldiv)?"ldiv":"lrem"));
      fflush(stdout);

      goto signal_handler_error;
    }
  }	// if (sig == SIGFPE)
#endif	// ARITHEXC_BY_SIGNAL
#if defined(NULLEXC_BY_SIGNAL) || defined(ARITHEXC_BY_SIGNAL)
  else
#endif
  {
    return FALSE;
  }


#ifdef RUNTIME_DEBUG
  // examine whether the signal is pending
  {
    sigset_t pending_set;
    if ((sigpending(&pending_set)) < 0)  perror("sigpending");
    else {
      printf("signal %d is pending: %s\n",
	sig, (sigismember(&pending_set, sig) ? "TRUE":"FALSE"));
      fflush(stdout);
    }
  }
#endif


  // jump to exc_new or exc_handler in compiled code

#if 0
  // re-initialize signal handler
  // associated with raised signal
  {
    struct sigaction sigAct = sigActForHandler;
    sigaction(sig, &sigAct, (struct sigaction *)NULL);
  }

#ifdef __FreeBSD__
  // unmask the signal
  {
    sigset_t sigmask;
    sigemptyset(&sigmask);
    sigaddset(&sigmask, sig);
    sigprocmask(SIG_UNBLOCK, &sigmask, NULL);
  }
#endif	// __FreeBSD__
#endif	// 0


  // set (int32_t)bytepcoff (local variable in assembledCode)
  *(((int32_t *)sc->SC_EBP) - 1) = tentry->byteoff;


  {
    uint32_t exc_handler;
    exc_handler = jump_target ? jump_target :
		(uint32_t)(mb->CompiledCode + codeinfo->exc_handler_nativeoff);
#ifdef RUNTIME_DEBUG
    printf("jump to 0x%x\n", exc_handler);  fflush(stdout);
#endif
    sc->SC_EIP = exc_handler;
  }

  SIGRETURN(sc, ebp);
#endif	// EXC_BY_SIGNAL
#endif	// EXC_BY_SIGNAL || GET_SIGCONTEXT

  return FALSE;

signal_handler_error:
  fprintf(stderr, "Signal %d\n", sig);
  fflush(stderr);
#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
  if (sc) {
    struct methodblock *sig_mb = methodByPC((unsigned char *)sc->SC_EIP);
    fprintf(stderr, "  in ");
    if (sig_mb) {
      uint32_t nativepc =
		(uint32_t)sc->SC_EIP - (uint32_t)sig_mb->CompiledCode;
      CodeInfo *info = (CodeInfo *)sig_mb->CompiledCodeInfo;
      fprintf(stderr, "%s#%s %s\n",
		cbName(fieldclass(&sig_mb->fb)),
		sig_mb->fb.name, sig_mb->fb.signature);
      fprintf(stderr, "  native PC: 0x%x(%d)\n", nativepc, nativepc);
#ifdef EXC_BY_SIGNAL
      if (info) {
	throwentry *tentry = throwtableGet(info, nativepc);
	if (tentry) {
	  int bytepc = tentry->byteoff;
	  fprintf(stderr, "  byte   PC: 0x%x(%d)\n", bytepc, bytepc);
	}
      }
#endif	// EXC_BY_SIGNAL
    }
    {
      char fpuenv[28];
      fillFPUEnv(fpuenv);
      showFPUEnv(stderr, "  ", fpuenv);
    }
    fprintf(stderr, "\n");
    fflush(stderr);

    showSigcontext(sc);

    fprintf(stderr, "\n");
    fflush(stderr);
  }
#endif
  JVM_Exit(1);
}


static void showFPUEnv(FILE *fp, char *indent, char fpuenv[28]) {
  if (!indent)  indent = "";
  if (!fp)  fp = stdout;

  fprintf(fp, "%sFPU control word: 0x%04x\n", indent,
		*((int16_t *)(fpuenv + 0)));
  fprintf(fp, "%sFPU status  word: 0x%04x\n", indent,
		*((int16_t *)(fpuenv + 4)));
  fprintf(fp, "%sFPU tag     word: 0x%04x\n", indent,
		*((int16_t *)(fpuenv + 8)));
  fprintf(fp, "%sFPU FIP         : 0x%08x\n", indent,
		*((int32_t *)(fpuenv + 12)));

  fflush(fp);
}


#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
/*
 * show contents of structure sigcontext
 */
void showSigcontext(SIGCONTEXT *sc) {
  printf("SS: %04x, CS: %04x, DS: %04x, ES: %04x, FS: %04x, GS: %04x\n",
#if defined(__FreeBSD__)
	sc->sc_ss & 0xffff, sc->sc_cs & 0xffff,
	sc->sc_ds & 0xffff, sc->sc_es & 0xffff,
	sc->sc_fs & 0xffff, sc->sc_gs & 0xffff
#else
	sc->ss, sc->cs, sc->ds, sc->es, sc->fs, sc->gs
#endif
  );

  printf("EAX: %08x, ECX: %08x, EDX: %08x, EBX: %08x\n",
	sc->SC_EAX, sc->SC_ECX, sc->SC_EDX, sc->SC_EBX);

  printf("ESI: %08x, EDI: %08x\n",
	sc->SC_ESI, sc->SC_EDI);

  printf("ESP: %08x, EBP: %08x EIP: %08x\n",
	sc->SC_ESP, sc->SC_EBP, sc->SC_EIP);

#if defined(linux)
  printf("ESP at signal: %08x\n", sc->esp_at_signal);
#endif

  {
    uint32_t *esp = (uint32_t *)sc->SC_ESP;
    if (esp)
      printf("(ESP+4): %08x, (ESP): %08x\n", *(esp+1), *esp);
  }
  {
    unsigned char *eip = (unsigned char *)sc->SC_EIP;
    if (eip)
      printf("(EIP): %02x %02x %02x %02x %02x %02x %02x %02x"
		"  %02x %02x %02x %02x %02x %02x %02x %02x\n",
	*eip,*(eip+1),*(eip+2),*(eip+3),*(eip+4),*(eip+5),*(eip+6),*(eip+7),
	*(eip+8),*(eip+9),*(eip+10),*(eip+11),*(eip+12),*(eip+13),
	*(eip+14),*(eip+15));
  }

  printf("trapno: %02x\n", sc->SC_TRAPNO);

  fflush(stdout);
}
#endif	// EXC_BY_SIGNAL || GET_SIGCONTEXT
