/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999,2000 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

#include <stdlib.h>	// for getenv(),atoi()
#include <sys/stat.h>	// for gdbm_open(), stat()
#include <unistd.h>	// for stat()
#include <string.h>	// for memset()

#include "compiler.h"

#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
#  include <signal.h>
#endif	// EXC_BY_SIGNAL || GET_SIGCONTEXT

#if JDK_VER >= 12
#  include "jit.h"	// for JITInterface, JITInterface6
#endif

#ifdef CODE_DB
#  include <fcntl.h>	// for O_...
#  include <dlfcn.h>	// for dl*()
#endif	// CODE_DB

#ifdef METAVM
#  include "jni.h"
#  include "metavm/metavm.h"	// for macro METAVM_PKG
#endif


//
// Global Variables
//
#ifndef IGNORE_DISABLE
bool_t compiler_enabled = TRUE;
#endif	// IGNORE_DISABLE

// for strictfp
bool_t is_fpupc_double = FALSE;

#ifdef CODE_DB
#  ifdef GDBM
#    include <gdbm.h>
GDBM_FILE db = NULL;
#  else
#    include <ndbm.h>
DBM *db = NULL;
#  endif
int db_page = -1;
#endif	// CODE_DB

struct bool_opt_entry { char *name; int num; };
struct bool_opt_entry bool_opt_entry[] = {
  {"quiet", OPT_QUIET},
  {"outcode", OPT_OUTCODE},
  {"codesize", OPT_CODESIZE},
  {"dontcmplvmclz", OPT_DONTCMPLVMCLZ},
  {"igndisable", OPT_IGNDISABLE},
#ifdef CODE_DB
  {"codedb", OPT_CODEDB},
#endif	// CODE_DB
  {"cmplatload", OPT_CMPLATLOAD},
  {"ignlock", OPT_IGNLOCK},
  {"ignstrictfp", OPT_IGNSTRICTFP},
  {"frcstrictfp", OPT_FRCSTRICTFP},
  {NULL, -1}
};
int options = 0;
int opt_cmplthreshold = 0;

void *sym_compileAndInvokeMethod, *sym_invokeJITCompiledMethod;
void *sym_invokeJavaMethod, *sym_invokeSynchronizedJavaMethod;
void *sym_invokeAbstractMethod;
void *sym_invokeNativeMethod, *sym_invokeSynchronizedNativeMethod;
void *sym_invokeJNINativeMethod, *sym_invokeJNISynchronizedNativeMethod;
void *sym_invokeLazyNativeMethod;
#ifdef CODE_DB
#  ifdef GDBM
GDBM_FILE (*sym_dbm_open)(char *,int,int,int,void (*)());
void (*sym_dbm_close)(GDBM_FILE);
int (*sym_dbm_store)(GDBM_FILE,datum,datum,int);
datum (*sym_dbm_fetch)(GDBM_FILE,datum);
void (*sym_dbm_sync)(GDBM_FILE);
#  else
DBM *(*sym_dbm_open)(const char *,int,int);
void (*sym_dbm_close)(DBM *);
int (*sym_dbm_store)(DBM *,datum,datum,int);
datum (*sym_dbm_fetch)(DBM *,datum);
#  endif
#endif	// CODE_DB


//
// Local Functions
//
static void initializeClassHook(ClassClass *);
static void freeClass(ClassClass *);
static unsigned char *compiledCodePC(JavaFrame *, struct methodblock *);
static bool_t pcInCompiledCode(unsigned char *pc, struct methodblock *);
#ifdef DIRECT_INVOCATION
static JavaFrame *framePrev(JavaFrame *frame, JavaFrame *frame_buf);
#endif	// DIRECT_INVOCATION
#if JDK_VER >= 12
static void *frameID(JavaFrame *frame);
#endif
#ifndef IGNORE_DISABLE
static void compilerEnable();
static void compilerDisable();
#endif	// IGNORE_DISABLE
static bool_t compileClass(ClassClass *);
static bool_t compileClasses(Hjava_lang_String *);
static HObject *compilerCommand(HObject *);


/*
 * Initialization of the compiler.
 * This method is called by JVM.
 */
void java_lang_Compiler_start(
#if JDK_VER < 12
  void ** vector
#else
  JITInterface *jitinterface
#endif
) {
  ExecEnv *ee = EE();
  MONITOR_T key;
  int version;
  int compilerVersion;

#ifdef COMPILE_DEBUG
  printf("java_lang_Compiler_start() called.\n");
  fflush(stdout);
#endif

  // prevent unloading of java.lang.Compiler class
  {
    ClassClass *compiler_cb = FindClass(ee, "java/lang/Compiler", TRUE);
    if (!compiler_cb) {
      printf("FATAL: cannot find the java.lang.Compiler class.\n");
      JVM_Exit(1);
    }
    CCSet(compiler_cb, Sticky);
  }


  // version check

#if JDK_VER < 12
  version = *(int *)vector[0];
#else
  version = *(jitinterface->JavaVersion);
#endif
  compilerVersion = version >> 24;

  if (compilerVersion != COMPILER_VERSION) {
    printf("warning: version num. of compiler interface is not %d: %d\n",
	COMPILER_VERSION, compilerVersion);
  }

#ifdef COMPILE_DEBUG
  {
    int javaVersion, javaMinorVersion;
    javaVersion = (version >> 16) & 0xff;
    javaMinorVersion = version & 0xffff;

    printf("Version num. of class file format: %d.%d\n",
	javaVersion, javaMinorVersion);
  }
#endif	// COMPILE_DEBUG


  // get options
  {
    char *opts = getenv("JAVA_COMPILER_OPT");
    char *opt;

    if (opts) {
#ifdef RUNTIME_DEBUG
      printf("  JAVA_COMPILER_OPT: %s\n", opts);
      fflush(stdout);
#endif

      opt = strtok(opts, ", ");
      while (opt) {
	int i = 0;
	char *opt_name;
	while (opt_name = bool_opt_entry[i].name, opt_name) {
	  if (!strcmp(opt, opt_name)) {
	    OPT_SET(bool_opt_entry[i].num);
	    if (!OPT_SETQ(OPT_QUIET))  printf(" option: %s\n", opt_name);
	  }
	  i++;
	}
	if (!strncmp(opt, "cmplthreshold=", 14)) {
	  opt_cmplthreshold = atoi(opt+14);
	  if (!OPT_SETQ(OPT_QUIET))
	    printf(" option: cmplthreshold = %d\n", opt_cmplthreshold);
	}
	opt = strtok(NULL, ", ");
      }

      fflush(stdout);
    }
  }


  // show credit
  if (!OPT_SETQ(OPT_QUIET)) {
    fprintf(stderr, CREDIT);
    fflush(stderr);
  }


#ifdef FORCE_DOUBLE_PRECISION
  // set rounding precision as double
  {
    unsigned short cw;
    asm("fnstcw %0" : "=m"(cw));
    cw = (cw & ~0x0300) | 0x0200;
    asm("fldcw %0" : : "m"(cw));
  }
#endif	// FORCE_DOUBLE_PRECISION


  // prohibit generating lossy opcodes
  UseLosslessQuickOpcodes = TRUE;


  // resolve some symbols
#define RESOLVE_A_INVOKER(NAME)	\
	if (!(sym_##NAME = (void *)symbolInSystemClassLoader(#NAME))) {\
		printf("FATAL: cannot resolve a symbol: %s\n", #NAME);\
		JVM_Exit(1);\
	}

  RESOLVE_A_INVOKER(compileAndInvokeMethod);
  RESOLVE_A_INVOKER(invokeJITCompiledMethod);
  RESOLVE_A_INVOKER(invokeJavaMethod);
  RESOLVE_A_INVOKER(invokeSynchronizedJavaMethod);
  RESOLVE_A_INVOKER(invokeAbstractMethod);
  RESOLVE_A_INVOKER(invokeNativeMethod);
  RESOLVE_A_INVOKER(invokeSynchronizedNativeMethod);
  RESOLVE_A_INVOKER(invokeJNINativeMethod);
  RESOLVE_A_INVOKER(invokeJNISynchronizedNativeMethod);
  RESOLVE_A_INVOKER(invokeLazyNativeMethod);

#ifdef COMPILE_DEBUG
#  define SHOW_A_INVOKER(NAME) \
  printf("  " #NAME ":\t0x%08x\n", (int)sym_##NAME)

  printf("symbols:\n");
  SHOW_A_INVOKER(compileAndInvokeMethod);
  SHOW_A_INVOKER(invokeJITCompiledMethod);
  SHOW_A_INVOKER(invokeJavaMethod);
  SHOW_A_INVOKER(invokeSynchronizedJavaMethod);
  SHOW_A_INVOKER(invokeAbstractMethod);
  SHOW_A_INVOKER(invokeNativeMethod);
  SHOW_A_INVOKER(invokeSynchronizedNativeMethod);
  SHOW_A_INVOKER(invokeJNINativeMethod);
  SHOW_A_INVOKER(invokeJNISynchronizedNativeMethod);
  SHOW_A_INVOKER(invokeLazyNativeMethod);
#endif


  // initialize all classes already loaded
  if (!OPT_SETQ(OPT_DONTCMPLVMCLZ)) {
    ClassClass **clazzptr;
    int i;

#if JDK_VER >= 12
    BINCLASS_LOCK(sysThreadSelf());
#else
    BINCLASS_LOCK();
#endif
#ifdef COMPILE_DEBUG
    printf("%d classes is already loaded.\n", nbinclasses);
    fflush(stdout);
#endif

    // initialize
    for (i = nbinclasses, clazzptr = binclasses; --i >= 0; clazzptr++) {
#ifdef COMPILE_DEBUG
      printf("init. a class in VM: %x,%s\n", *clazzptr, cbName(*clazzptr));
      fflush(stdout);
#endif
      initializeClassForJIT(*clazzptr, TRUE);
    }

    if (OPT_SETQ(OPT_CMPLATLOAD)) {
      // compile
      ClassClass **classes;
      classes = (ClassClass **)sysMalloc(sizeof(ClassClass *) * nbinclasses);
      memcpy(classes, binclasses, sizeof(ClassClass *) * nbinclasses);

      clazzptr = classes;
      for (i = nbinclasses, clazzptr = classes; --i >= 0; clazzptr++) {
#ifdef COMPILE_DEBUG
	printf("compile a class in VM: %x,%s\n", *clazzptr, cbName(*clazzptr));
	fflush(stdout);
#endif
	compileClass(*clazzptr);
      }

      sysFree(classes);
    }

#if JDK_VER >= 12
    BINCLASS_UNLOCK(sysThreadSelf());
#else
    BINCLASS_UNLOCK();
#endif
  }


  // initialize for signal handler
#if (defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)) && defined(SEARCH_SIGCONTEXT)
#  if JDK_VER < 12
  *(char **)vector[3] = (char *)examineSigcontextNestCount;
#  else
  *((JITInterface6 *)jitinterface)->p_CompiledCodeSignalHandler =
	examineSigcontextNestCount;
#  endif

  asm(".global exam_point\n\t"
      "exam_point: int $0x10");

#  if JDK_VER < 12
  *(char **)vector[3] = NULL;
#  else
  *((JITInterface6 *)jitinterface)->p_CompiledCodeSignalHandler = NULL;
#  endif

  {
    if (sc_nest < 0) {
      printf("FATAL: cannot examine the offset of signal context.\n");
      JVM_Exit(1);
    }
  }
#endif	// SEARCH_SIGCONTEXT


  // set up link vector
#if JDK_VER < 12
  *(char **)vector[1] = (char *)initializeClassHook;
  *(char **)vector[2] = (char *)sym_invokeJITCompiledMethod;
  *(char **)vector[3] = (char *)signalHandler;
  *(char **)vector[4] = (char *)freeClass;
  *(char **)vector[5] = (char *)compileClass;
  *(char **)vector[6] = (char *)compileClasses;
#ifndef IGNORE_DISABLE
  if (!OPT_SETQ(OPT_IGNDISABLE)) {
    *(char **)vector[7] = (char *)compilerEnable;
    *(char **)vector[8] = (char *)compilerDisable;
  }
#endif	// IGNORE_DISABLE
  *(char **)vector[10] = (char *)pcInCompiledCode;
  *(char **)vector[11] = (char *)compiledCodePC;
#ifdef DIRECT_INVOCATION
  *(char **)vector[70] = (char *)framePrev;
#endif	// DIRECT_INVOCATION
#ifdef COMPILE_DEBUG
  {
    int i;
    for (i = 1; i <= 8; i++)
      printf("*vector[%d]: 0x%08x\n", i, (int)*(char **)vector[i]);
    printf("*vector[10]: 0x%08x\n", i, (int)*(char **)vector[10]);
    printf("*vector[11]: 0x%08x\n", i, (int)*(char **)vector[11]);
    printf("*vector[70]: 0x%08x\n", i, (int)*(char **)vector[70]);
    fflush(stdout);
  }
#endif	// COMPILE_DEBUG
#else	// JDK_VER
  {
    JITInterface6 *ji6 = (JITInterface6 *)jitinterface;

    *ji6->p_InitializeForCompiler = initializeClassHook;
    *ji6->p_invokeCompiledMethod = sym_invokeJITCompiledMethod;
    *ji6->p_CompiledCodeSignalHandler = signalHandler;
    *ji6->p_CompilerFreeClass = freeClass;

    *ji6->p_CompilerCompileClass = compileClass;
    *ji6->p_CompilerCompileClasses = compileClasses;
#ifndef IGNORE_DISABLE
    if (!OPT_SETQ(OPT_IGNDISABLE)) {
      *ji6->p_CompilerEnable = compilerEnable;
      *ji6->p_CompilerDisable = compilerDisable;
    }
#endif	// IGNORE_DISABLE

    *ji6->p_PCinCompiledCode = pcInCompiledCode;
    *ji6->p_CompiledCodePC = compiledCodePC;
#ifdef DIRECT_INVOCATION
    *ji6->p_CompiledFramePrev = framePrev;
#endif	// DIRECT_INVOCATION
    *ji6->p_CompiledFrameID = frameID;	// only for JDK >= 1.2
  }
#endif	// JDK_VER


#ifdef CODE_DB
  // open DB and page file
  if (OPT_SETQ(OPT_CODEDB)) {
    void *dl_dbm;
    if (!(dl_dbm = dlopen(LIBDBM, RTLD_LAZY))) {
      fputs(dlerror(), stderr);  fputc('\n', stderr);
      fprintf(stderr, "failed to open " LIBDBM ".\n");
      goto codedb_init_fail;
    }
#ifdef GDBM
    sym_dbm_open = (GDBM_FILE (*)(char *,int,int,int,void (*)()))
	dlsym(dl_dbm, "gdbm_open");
    sym_dbm_close = (void (*)(GDBM_FILE))dlsym(dl_dbm, "gdbm_close");
    sym_dbm_store = (int (*)(GDBM_FILE,datum,datum,int))
	dlsym(dl_dbm, "gdbm_store");
    sym_dbm_fetch = (datum (*)(GDBM_FILE,datum))dlsym(dl_dbm, "gdbm_fetch");
    sym_dbm_sync = (void (*)(GDBM_FILE))dlsym(dl_dbm, "gdbm_sync");
#else
    sym_dbm_open = (DBM *(*)(const char *,int,int))dlsym(dl_dbm, "dbm_open");
    sym_dbm_close = (void (*)(DBM *))dlsym(dl_dbm, "dbm_close");
    sym_dbm_store = (int (*)(DBM *,datum,datum,int))dlsym(dl_dbm, "dbm_store");
    sym_dbm_fetch = (datum (*)(DBM *,datum))dlsym(dl_dbm, "dbm_fetch");
#endif
    if (!(((int32_t)sym_dbm_open) && ((int32_t)sym_dbm_close) &&
	  ((int32_t)sym_dbm_store) && ((int32_t)sym_dbm_fetch))
#ifdef GDBM
	  && ((int32_t)sym_dbm_sync)
#endif
	) {
      fprintf(stderr, "cannot get symbols to handle DBM.\n");
      goto codedb_init_fail;
    }

    if ((db_page = open(CODEDB_PAGE,
		O_RDWR|O_CREAT, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH)) < 0) {
      perror("open");  goto codedb_init_fail;
    }

#ifdef GDBM
    if (!(db = sym_dbm_open(CODEDB_DB, 512,
	GDBM_WRCREAT, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH, NULL))) {
      perror("gdbm_open");  goto codedb_init_fail;
    }
#else
    if (!(db = sym_dbm_open(CODEDB_DB, O_RDWR|O_CREAT, 0))) {
      perror("dbm_open");  goto codedb_init_fail;
    }
#endif

    goto codedb_init_done;
  codedb_init_fail:
    fprintf(stderr, "disable codedb.\n");  OPT_RESET(OPT_CODEDB);
    if (db_page >= 0)  close(db_page);
    JVM_Exit(1);
  codedb_init_done:
  }
#endif	// CODE_DB


#ifdef METAVM
  {
    ExecEnv *ee = EE();
    // force JVM to load Proxy class
    FindClass(ee, METAVM_PKG "Proxy", TRUE);

    // set remote flag on
    REMOTE_FLAG_ON(ee);
  }
#endif


  // for strictfp
  // check the FPU roundig precision
  {
    uint16_t cw;
    asm("fstcw %0" : "=m" (cw));
    is_fpupc_double = ((cw & 0x0300) == 0x0200);
#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
    printf("FPU rounding precision: %sdouble.\n",
	(is_fpupc_double?"":"not "));
#endif
  }


#ifdef COMPILE_DEBUG
  printf("java_lang_Compiler_start() done.\n");
  fflush(stdout);
#endif
}


/*
 * Initialize the class at the time one is loaded.
 */
void initializeClassForJIT(ClassClass *cb, bool_t initInvoker) {
  struct methodblock *mb;
  int mb_count;

#ifdef METAVM
  // force a loaded class to implement java.io.Serializable
  {
    ExecEnv *ee = EE();  JNIEnv *env = EE2JNIEnv(ee);
    static ClassClass *clz_Serializable = NULL;
    if (!clz_Serializable) {
      jclass jclz_Ser = (*env)->FindClass(env, "java/io/Serializable");
      (*env)->NewGlobalRef(env, jclz_Ser);
      clz_Serializable = DeRef(env, jclz_Ser);
    }

    if (!cbIsInterface(cb) && (cb != classJavaLangObject))
      forceToImplement(ee, cb, clz_Serializable);
  }
#endif

  mb = cbMethods(cb);
  mb_count = cbMethodsCount(cb);
  for (; mb_count-- > 0; mb++) {
    int access = mb->fb.access;

    if (access & ACC_ABSTRACT)  continue;

    // initialize mb->CompiledCodeInfo
    if (!(mb->CompiledCodeInfo))
      if (!prepareCompiledCodeInfo(EE(), mb))  continue;

    if (access & ACC_NATIVE)  continue;

    if (!initInvoker)  continue;

    // initialize invoker
    if ((OPT_SETQ(OPT_CMPLCLINIT) || strcmp(mb->fb.name, "<clinit>")) &&
	(mb->invoker != sym_compileAndInvokeMethod) &&	// not init.ed yet
	(mb->invoker != sym_invokeJITCompiledMethod)) {	// not compiled yet
      mb->invoker = sym_compileAndInvokeMethod;
      mb->CompiledCode = NULL;
    }
  }
}


//
// Local Functions
//

static void initializeClassHook(ClassClass *cb) {
#ifdef COMPILE_DEBUG
  printf("initializeClassHook(%s) called.\n", cbName(cb));
  fflush(stdout);
#endif

  initializeClassForJIT(cb, TRUE);

  if (OPT_SETQ(OPT_CMPLATLOAD)) {	// have to done after initialization
    compileClass(cb);
  }

#ifdef COMPILE_DEBUG
  printf("initializeClassHook(%s) done.\n", cbName(cb));
  fflush(stdout);
#endif
}


/*
 * Freeing class stuffs related to the compiler.
 */
static void freeClass(ClassClass *cb) {
  struct methodblock *mb = cbMethods(cb);
  struct methodblock *mb_end = mb + cbMethodsCount(cb);

#ifdef COMPILE_DEBUG
  printf("freeClass() called.\n");
  fflush(stdout);
#endif

  for (; mb < mb_end; mb++)
    freeMethod(mb);
}


static unsigned char *compiledCodePC(JavaFrame *frame,struct methodblock *mb) {
#ifdef DIRECT_INVOCATION
  // in case of DIRECT_INVOCATION, frame->lastpc is not up-to-date.
  return mb->code;
#else
  return (frame ? frame->lastpc : mb->code);
#endif	// DIRECT_INVOCATION
}


static bool_t pcInCompiledCode(
	unsigned char *pc /* mb->lastpc */, struct methodblock *mb) {
#if 1	// mb->lastpc is a pc on bytecode
  unsigned int off = pc - mb->code;
  return (off < mb->code_length);
#else	// mb->lastpc is a pc on compiled code
  CodeInfo *info = (CodeInfo *)mb->CompiledCodeInfo;
  if (info) {
    unsigned char *compiled_code = (unsigned char *)mb->CompiledCode;
    unsigned char *limit = compiled_code + info->code_size;
    if (compiled_code != limit) {
      if ((compiled_code <= pc) && (pc < limit))  return TRUE;
    }
  }
  return FALSE;
#endif
}


#ifdef DIRECT_INVOCATION
#undef FRAMEPREV_DEBUG
static JavaFrame *framePrev(JavaFrame *frame, JavaFrame *frame_buf) {
  uint32_t *basep;
  unsigned char *ret_addr;
  extern void invokeJIT_compiled_done(), candi_compiled_done();

#ifdef FRAMEPREV_DEBUG
{
  struct methodblock *mb;
  printf("	| framePrev() called.\n");
  mb = frame->current_method;
  printf("	|   %s#%s %s\n",
	cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
  fflush(stdout);
}
#endif
  basep = (uint32_t *)(frame->vars);
#ifdef FRAMEPREV_DEBUG
printf("	| basep: %x\n", basep);
fflush(stdout);
#endif

  if (frame != frame_buf) {	// first time
    memset((void *)frame_buf, 0, sizeof(JavaFrame));
    frame_buf->prev = frame->prev;
  }

  ret_addr = (unsigned char *)basep[1];
  if ((ret_addr == (unsigned char *)invokeJIT_compiled_done) ||
      (ret_addr == (unsigned char *)candi_compiled_done)) {
#ifdef FRAMEPREV_DEBUG
printf("	| limit.\n\n");
fflush(stdout);
#endif
    return frame->prev;
  }

  basep = (uint32_t *)basep[0];
  frame_buf->vars = (stack_item *)basep;
  frame_buf->current_method = (struct methodblock *)basep[3];
#ifdef FRAMEPREV_DEBUG
{
  struct methodblock *mb;
  mb = frame_buf->current_method;
  printf("	| method: %x\n", mb);
  fflush(stdout);
  printf("	|   buf->cur_method: %s#%s %s\n\n",
	 cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
  fflush(stdout);
}
#endif
  return frame_buf;
}
#endif	// DIRECT_INVOCATION


#if JDK_VER >= 12
static void *frameID(JavaFrame *frame) {
  return (void *)frame->vars;
}
#endif



#ifndef IGNORE_DISABLE
/*
 * Enable and Disable the compiler.
 */
static void compilerEnable() {
#ifdef RUNTIME_DEBUG
  printf("compilerEnable() called.\n");  fflush(stdout);
#endif
  compiler_enabled = TRUE;
}

/*
 * Enable and Disable the compiler.
 */
static void compilerDisable() {
#ifdef RUNTIME_DEBUG
  printf("compilerDisable() called.\n");  fflush(stdout);
#endif
  compiler_enabled = FALSE;
}
#endif	// IGNORE_DISABLE


/*
 * Compile the class. Not only initialization.
 */
static bool_t compileClass(ClassClass *cb) {
  struct methodblock *mb, *mb_end;

#ifdef COMPILE_DEBUG
  printf("\n");
  printf("compileClass(%s) called.\n", cbName(cb));
  fflush(stdout);
#endif

#ifndef IGNORE_DISABLE
  if (!compiler_enabled) {
#ifdef COMPILE_DEBUG
    printf("  compiler has been disabled.\n  return.\n");
    fflush(stdout);
#endif
    return TRUE;
  }
#endif	// IGNORE_DISABLE

  mb = cbMethods(cb);
  mb_end = mb + cbMethodsCount(cb);
  for (; mb < mb_end; mb++) {
    int access = mb->fb.access;

    if (access & ACC_ABSTRACT)  continue;

    // initialize mb->CompiledCodeInfo
    if (!(mb->CompiledCodeInfo))  prepareCompiledCodeInfo(EE(), mb);

    if (access & ACC_NATIVE)  continue;

    if (compileMethod(mb))  return FALSE;
  }

#ifdef COMPILE_DEBUG
  printf("compileClass(%s) done.\n", cbName(cb));
  fflush(stdout);
#endif

  return TRUE;
}


/*
 * Compile the classes specified by comma-delimited names.
 */
static bool_t compileClasses(Hjava_lang_String *nameStr) {
  char *names = allocCString(nameStr);
  char *name = names;
  char *p = names;
  bool_t done = FALSE;

#ifdef COMPILE_DEBUG
  printf("compileClasses() called.\n");
  fflush(stdout);
#endif

  if (names) {
    ClassClass *cb;

    do {
      while ((*p != ',') && (*p != '\0'))  p++;
      if (*p == '\0')  done = TRUE;
      else  *p = '\0';

#ifdef COMPILE_DEBUG
      printf("compileClasses(): classname is `%s'\n", name);
      fflush(stdout);
#endif
      cb = FindClass(NULL, name, TRUE);
      if (cb)
	if (!(compileClass(cb)))  goto compileDone;

      p++;
      name = p;
    } while (!done);

  compileDone:
    sysFree(names);

    return TRUE;
  }

  return FALSE;
}


/*
 * Process the command from Java code.
 */
static HObject *compilerCommand(HObject *obj) {
  HObject *ret = NULL;
  
#ifdef COMPILE_DEBUG
  printf("compilerCommand() called.\n");
  fflush(stdout);
#endif

  // do nothing.

  return ret;
}
