/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

package NET.shudo.metavm;

import java.util.Dictionary;
import java.util.Hashtable;


public class TypeUtil {
  private TypeUtil() {}	// indicates prohibition of instantiation


  // types corresponded with argument of JVM op. `newarray'
  public static final int T_OBJECT	= 0;
  public static final int T_BOOLEAN	= 4;
  public static final int T_CHAR	= 5;
  public static final int T_FLOAT	= 6;
  public static final int T_DOUBLE	= 7;
  public static final int T_BYTE	= 8;
  public static final int T_SHORT	= 9;
  public static final int T_INT		= 10;
  public static final int T_LONG	= 11;
  public static final int T_MAX		= 11;


  private static Dictionary primSigTable;
  static {
    primSigTable = new Hashtable();
    primSigTable.put("void", "V");
    primSigTable.put("boolean", "Z");
    primSigTable.put("byte", "B");
    primSigTable.put("char", "C");
    primSigTable.put("short", "S");
    primSigTable.put("int", "I");
    primSigTable.put("long", "J");
    primSigTable.put("float", "F");
    primSigTable.put("double", "D");
  }

  private static Class[] primTypeByCode;
  static {
    primTypeByCode = new Class[T_MAX + 1];
    primTypeByCode[T_BOOLEAN] = boolean.class;
    primTypeByCode[T_CHAR] = char.class;
    primTypeByCode[T_FLOAT] = float.class;
    primTypeByCode[T_DOUBLE] = double.class;
    primTypeByCode[T_BYTE] = byte.class;
    primTypeByCode[T_SHORT] = short.class;
    primTypeByCode[T_INT] = int.class;
    primTypeByCode[T_LONG] = long.class;
  }


  public static Class primTypeByCode(int tcode) {
    Class ret = null;

    try {
      ret = primTypeByCode[tcode];
    }
    catch (ArrayIndexOutOfBoundsException e) {
      // not reached.
      System.err.println("type code is invalid: " + tcode);
    }

    return ret;
  }


  public static Class arrayType(Class clazz) {
    ClassLoader cl = clazz.getClassLoader();
    String name = clazz.getName();
    String componentName = null;
    String aryClazzName = null;
    Class ret;

    if (name.startsWith("["))  componentName = name;
    else {
      componentName = (String)primSigTable.get(name);
      if (componentName == null) {
	componentName = "L" + name + ";";
      }
    }
    aryClazzName = "[" + componentName;

    try {
      if (cl == null)
	ret = Class.forName(aryClazzName);
      else
	ret = cl.loadClass(aryClazzName);
    }
    catch (ClassNotFoundException e) {
      // not reached.
      System.err.println("class object of array of `" + name +
				"' is not obtained.");
      ret = null;
    }

    return ret;
  }


  // these methods must be protected, now testing...
  synchronized public static native void addCheckPassType(Class clazz);
  synchronized public static native void clearCheckPassType();

  public static void forceToImplement(Class clazz, Class intf) {
    if (intf.isAssignableFrom(clazz))  return;
    forceToImplement0(clazz, intf);
  }
  private static native void forceToImplement0(Class clazz, Class intf);


  // for test
  public static void main(String[] args) {
    Class clazz = null;

    clazz = int.class;
    clazz = arrayType(clazz);
    System.out.println(clazz.getName());

    clazz = int[].class;
    clazz = arrayType(clazz);
    System.out.println(clazz.getName());

    clazz = java.lang.Object.class;
    clazz = arrayType(clazz);
    System.out.println(clazz.getName());

    try {
      clazz = Class.forName("[Ljava.lang.Object;");
	// This is array class, so load from local disk.
    }
    catch (ClassNotFoundException e) {
      e.printStackTrace();
    }
    clazz = arrayType(clazz);
    System.out.println(clazz.getName());
  }
}
