/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

package NET.shudo.metavm;

import java.net.Socket;
import java.net.ServerSocket;
import java.io.IOException;


/**
 * This is MetaVM server which accepts incoming connection
 * and invokes request server.
 */
public class MetaVMServer implements Runnable {
  private static final String PROGRAM = "metavm";

  /** debug flag */
  private boolean debug = false;
  /** quiet flag */
  private boolean quiet = false;

  private ExpirationDaemon expDaemon;
  private ClassDistributionDaemon clzDaemon;


  /** thread pool */
  private ThreadPool pool = new ThreadPool(
    new ThreadPoolHook() {
      public void runHook(Thread t) { MetaVM.remoteTransparency(t, true); }
      public void doneHook(Thread t) { MetaVM.remoteTransparency(t, false); }
      public void newHook(Thread t) {}
    },
    MetaVM.debug
  );


  protected MetaVMServer() {}


  private void usage() {
    System.out.println("usage: " + this.PROGRAM + " [-h] [-d] [-b <num>]");
    System.out.println("\t-h:  Show this help message.");
    System.out.println("\t-p <num>:  specifies port number.");
    System.out.println("\t-n:  Set TCP_NODELAY to socket.");
    System.out.println("\t-b <num>:  specifies buffer size of BufferedStream in KB.");
    System.out.println("\t-l:  Always load classes from local disk.");
    System.out.println("\t-q:  Quiet.");
    System.out.println("\t-d:  For debug.");
  }


  private void start(String[] args) {
    // load the class MetaVM
    MetaVM.aServerExists = true;

    // parse options
    for (int i = 0; i < args.length; i++) {
      if (args[i].equals("-d")) {
	MetaVM.debug = true;
      }
      else if (args[i].equals("-p")) {
	String s = null;
	try { s = args[++i]; }
	catch (ArrayIndexOutOfBoundsException e) {
	  System.out.println("-p option require an argument.");
	  break;
	}

	int port = Integer.parseInt(s);
	if (port <= 0) {
	  System.out.println(
		"port number must be positive and non-zero number.");
	  break;
	}
	VMAddress.localPort(port);
	if (debug)
	  System.out.println("port number:" + VMAddress.localPort());
      }
      else if (args[i].equals("-n")) {
	MetaVM.tcp_nodelay = true;
      }
      else if (args[i].equals("-b")) {
	String s = null;
	try { s = args[++i]; }
	catch (ArrayIndexOutOfBoundsException e) {
	  System.out.println("-b option require an argument.");
	  break;
	}

	int size = Integer.parseInt(s);
	if (size < 0)  size = 0;
	MetaVM.bufsize = size;
	if (debug)
	  System.out.println("buffer size(KB): " + MetaVM.bufsize);
      }
      else if (args[i].equals("-l")) {
	MetaVM.load_class_locally = true;
      }
      else if (args[i].equals("-q")) {
	this.quiet = true;
      }
      else if (args[i].equals("-h") || args[i].equals("-?")) {
	usage();  System.exit(0);
      }
    }

    run();
  }


  public void run() {
    // set debug flag of this instance
    this.debug = MetaVM.debug;

    // invoke daemons
    Runnable r;
    Thread t;

    // invoke ExpirationDaemon
    r = expDaemon = new ExpirationDaemon();
    t = new Thread(r, "Expiration Daemon");
    t.setDaemon(true);	// daemon thread
    try {
      t.setPriority(Thread.MIN_PRIORITY);
    } catch (IllegalArgumentException e) {}
    t.start();


    // create a server socket
    ServerSocket servsock = null;
    int port = VMAddress.localPort();
    int retry = 10;

    while (retry > 0) {
      try {
	servsock = new ServerSocket(port);	// throws an exception
	if (debug) {
	  System.out.println("MetaVMServer: port no. is " + port);
	  System.out.flush();
	}

	VMAddress.localPort(port);
	break;
      }
      catch (IOException e) {
	if (debug)  e.printStackTrace();
	if (e instanceof java.net.BindException) {
	  if (debug) {
	    System.out.println("MetaVMServer port " + port + "/tcp is already in use.");
	    System.out.flush();
	  }
	  port++;  retry--;
	}
	else
	  System.exit(1);
      }
    }


    // invoke ClassDistributionDaemon
    r = clzDaemon = new ClassDistributionDaemon();
    t = new Thread(r, "Class Distribution Daemon");
    t.setDaemon(true);	// daemon thread
    try {
      t.setPriority(Thread.currentThread().getPriority() + 1);
    } catch (IllegalArgumentException e) {}
    t.start();



    // start up message
    if (!quiet) {
      System.out.print("  MetaVM server started on ");
      System.out.print(VMAddress.localAddress());
      System.out.println(".");
      System.out.flush();
    }


    // main loop
    while (true) {
      Socket sock = null;

      try {
	sock = servsock.accept();
	if (debug) {
	  System.out.println("MetaVMServer: accept.");
	}

	if (MetaVM.tcp_nodelay)
	  sock.setTcpNoDelay(true);
      }
      catch (Exception e) {
	if (debug)  e.printStackTrace();

	// to avoid busy loop
	try { Thread.sleep(2000); }
	catch (InterruptedException ie) {}

	continue;
      }

      Skeleton reqserv = null;
      try { reqserv = new Skeleton(this, sock); }
      catch (IOException e) {
	// not reached.
	System.err.println("initialization of Skeleton is failed.");
	continue;
      }

      pool.run(reqserv);
    }	// while (true)
  }


  protected void reset() {
    ExportTable.reset();
    RemoteClassLoader.reset();
    clzDaemon.reset();

    System.gc();
  }


  public static void main(String[] args) {
    System.runFinalizersOnExit(true);

    try {
      new MetaVMServer().start(args);
    }
    catch (UnsatisfiedLinkError e) {
      System.err.println("FATAL: MetaVM requires shuJIT (libmetavm.so).");
      e.printStackTrace();
      System.exit(1);
    }
  }
}
