/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

#include "compiler.h"

#ifdef METAVM
#include "metavm/metavm.h"	/* for GET_REMOTE_FLAG() */
#include "metavm/NET_shudo_metavm_Proxy_old.h"
#endif	/* METAVM */


#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
/*
 * show contents of structure sigcontext
 */
void showSigcontext(SIGCONTEXT *sc) {
  printf("EAX: %08x, ECX: %08x, EDX: %08x, EBX: %08x\n",
	sc->SC_EAX, sc->SC_ECX, sc->SC_EDX, sc->SC_EBX);

  printf("ESI: %08x, EDI: %08x\n",
	sc->SC_ESI, sc->SC_EDI);

  printf("ESP: %08x, EBP: %08x EIP: %08x\n",
	sc->SC_ESP, sc->SC_EBP, sc->SC_EIP);

#if defined(linux)
  printf("ESP at signal: %08x\n", sc->esp_at_signal);
#endif

  {
    uint32_t *esp = (uint32_t *)sc->SC_ESP;
    if (esp)
      printf("(ESP+4): %08x, (ESP): %08x\n", *(esp+1), *esp);
  }

  printf("SS: %04x, CS: %04x, DS: %04x, ES: %04x, FS: %04x, GS: %04x\n",
#if defined(__FreeBSD__)
	sc->sc_ss & 0xffff, sc->sc_cs & 0xffff,
	sc->sc_ds & 0xffff, sc->sc_es & 0xffff,
	sc->sc_fs & 0xffff, sc->sc_gs & 0xffff
#else
	sc->ss, sc->cs, sc->ds, sc->es, sc->fs, sc->gs
#endif
  );

  printf("trapno: %02x\n", sc->SC_TRAPNO);

  fflush(stdout);
}
#endif	/* EXC_BY_SIGNAL || GET_SIGCONTEXT */


/*
 * returns: return value size. -1 if an exception occurred.
 */
int invocationHelper(
#ifdef RUNTIME_DEBUG
	int runtime_debug,
#endif
	JHandle *obj, struct methodblock *method, int args_size, ExecEnv *ee,
	stack_item *var_base, int retsize, int bytepcoff
) {
  int access;
  bool_t (*invoker)(JHandle*,struct methodblock *,int,ExecEnv*);
  JavaFrame *cur_frame;	/* ee->current_frame */
#ifdef RUNTIME_DEBUG
  CodeInfo *info;
  info = (CodeInfo *)method->CompiledCodeInfo;
  if (!info) {
    printf("WARNING: method->CompiledCodeInfo is NULL. (invocationHelper())\n");
    fflush(stdout);
  }
#endif
  invoker = method->invoker;
  access = method->fb.access;

#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    printf("  invoker: %s (0x%x)\n",
	nameOfInvoker((void *)invoker), (int)invoker);
    fflush(stdout);
  }
#endif

  /* set ee->current_frame->lastpc */
  cur_frame = ee->current_frame;
  cur_frame->lastpc = cur_frame->current_method->code + bytepcoff;

#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    struct methodblock *cur_mb = cur_frame->current_method;
    char *sig = method->fb.signature;
    int i;

    printf("  local var base: 0x%x\n", (int)var_base);
    printf("    0x%x\n", *(int32_t *)var_base);
    printf("  ee: 0x%08x\n", (int)ee);
    printf("  current_frame: 0x%08x\n", (int)cur_frame);
    printf("  args_size, retsize, nlocals: %d, %d, %d\n",
		args_size, retsize, method->nlocals);
    printf("  obj: 0x%08x\n", (int)obj);
    printf("  method: 0x%08x\n", (int)method);
    printf("  caller method:");
    fflush(stdout);
    if (cur_mb)
      printf(" %s#%s %s",
		cbName(cur_mb->fb.clazz),
		cur_mb->fb.name, cur_mb->fb.signature);
    else
      printf(" (null)");
    fflush(stdout);
    printf(" (0x%08x)\n", (int)cur_mb);
    printf("  bytepcoff: 0x%x(%d)\n", bytepcoff, bytepcoff);
    fflush(stdout);

    printf("  obj: ");
    if (access & ACC_STATIC)
      printf("%s", unhand((Hjava_lang_Class *)obj)->name);
    else if (obj_flags(obj) == T_NORMAL_OBJECT)
      printf("%s", cbName(obj->methods->classdescriptor));
    else	/* array object */
      printf("array");
    printf(" (0x%08x)\n", (int)obj);
    printf("  method: %s#%s %s\n",
	cbName(fieldclass(&method->fb)), method->fb.name, sig);
    fflush(stdout);
    if (info) {
      printf("  ret val size: %d\n", info->ret_size);
      fflush(stdout);
    }
    printf("  acc: 0x%x", access);
    if (access & ACC_NATIVE)  printf(" native");
    if (access & ACC_MACHINE_COMPILED)  printf(" machine_compiled");
    if (access & ACC_PUBLIC)  printf(" public");
    if (access & ACC_PRIVATE)  printf(" private");
    if (access & ACC_PROTECTED)  printf(" protected");
    if (access & ACC_STATIC)  printf(" static");
    if (access & ACC_FINAL)  printf(" final");
    if (access & ACC_SYNCHRONIZED)  printf(" synchronized");
    if (access & ACC_ABSTRACT)  printf(" abstract");
    printf("\n");
    printf("  args_size: %d\n", args_size);
    fflush(stdout);
    i = 0;
    if (!(access & ACC_STATIC)) {
      printf("    0x%08x @ 0x%08x  ", (int)var_base[0].h, (int)var_base);
      fflush(stdout);
      showObjectBody("L;", var_base[0].h);
      printf("\n");
      i++;
    }
    if (sig[0] == '(')  sig++;
    for (; i < args_size; i++) {
      JHandle *arg = var_base[-i].h;
      if ((sig[0] == 'D') || (sig[0] == 'J'))  i++;
      printf("    0x%08x @ 0x%08x  ", (int)arg, (int)(var_base - i));
      fflush(stdout);
      sig = showObjectBody(sig, arg);
      printf("\n");
    }
    fflush(stdout);
  }
#endif	/* RUNTIME_DEBUG */


#if 1
#  ifdef RUNTIME_DEBUG
#    define CALL_INVOKER \
    asm("pushl 24(%ebp)");\
    asm("movl  %0,%%eax" : : "m" (invoker));\
    asm("pushl 20(%ebp)\n\tpushl 16(%ebp)\n\tpushl 12(%ebp)");\
    asm("call  *%eax\n\t"\
	"addl  $16,%esp");\
    { register int ret asm("eax");  if (!ret)  return -1; }
#  else
#    define CALL_INVOKER \
    asm("pushl 20(%ebp)");\
    asm("movl  %0,%%eax" : : "m" (invoker));\
    asm("pushl 16(%ebp)\n\tpushl 12(%ebp)\n\tpushl 8(%ebp)");\
    asm("call  *%eax\n\t"\
	"addl  $16,%esp");\
    { register int ret asm("eax");  if (!ret)  return -1; }
#  endif
#else
#  define CALL_INVOKER \
  if (!(invoker(obj, method, args_size, ee)))\
    return -1
	/* obj is required for enter/exit a monitor of the object. */
#endif

  cur_frame->returnpc = (unsigned char *)var_base;

  if (invoker == sym_invokeJITCompiledMethod) {
	/* Surely the compiled method is called */
    CALL_INVOKER;
  }
  else {
	/* normal Java or native method
	   or call compileAndInvokeMethod() */
    /* restack from native stack to JVM stack */
    stack_item *sp = var_base;
    stack_item *optop = cur_frame->optop;
    char *argsizes;

    argsizes = ((CodeInfo *)method->CompiledCodeInfo)->argsizes;
    while (*argsizes) {
      if (*argsizes == 1) { optop[0] = sp[0];  optop++;  sp--; }
      else { optop[0] = sp[-1];  optop[1] = sp[0];  optop += 2;  sp -= 2; }
      argsizes++;
    }

    CALL_INVOKER;

    if ((invoker == sym_invokeJavaMethod) ||
	(invoker == sym_invokeSynchronizedJavaMethod)) {
	/* normal Java method */
      int exec_ret;
      stack_item *old_optop;

#ifdef RUNTIME_DEBUG
      printf("call ExecuteJava(runtime.c): %s#%s.\n",
	cbName(method->fb.clazz), method->fb.name);  fflush(stdout);
#endif
      exec_ret = ExecuteJava(method->code, ee);
#ifdef RUNTIME_DEBUG
      printf("ExecuteJava(runtime.c) done: %s#%s.\n",
	cbName(method->fb.clazz), method->fb.name);  fflush(stdout);
#endif
      cur_frame = ee->current_frame;	/* reload */
      old_optop = cur_frame->optop;
#if defined(EXECUTEJAVA_IN_ASM) && (JDK_VER < 12)
	if (cur_frame->monitor) {
	  monitorExit2(ee, (MONITOR_T)cur_frame->monitor);
	}
#endif	/* EXECUTEJAVA_IN_ASM */
      cur_frame = cur_frame->prev;
      ee->current_frame = cur_frame;
      if (!exec_ret) {
#ifdef RUNTIME_DEBUG
	printf("  ExecuteJava() returned false.\n");
	if (exceptionOccurred(ee))
	  printf("  clazz of exc: %s\n",
		cbName(ee->exception.exc->methods->classdescriptor));
	fflush(stdout);
#endif
	return -1;
      }

#ifdef EXECUTEJAVA_IN_ASM
      /* This operation is required
	 only with x86 assembly ver. of executeJava.c */
      {
	stack_item *optop;
	if (retsize != 0) {
	  optop = cur_frame->optop;
	  if (retsize == 1) { optop[0] = old_optop[-1];  optop++; }
	  else {	/* retsize == 2 */
	    optop[0] = old_optop[-2];  optop[1] = old_optop[-1];  optop += 2;
	  }
	  cur_frame->optop = optop;
	}
      }
#endif
    }	/* normal Java method */
  }	/* normal Java or native method or compileAndInvokeMethod() */

#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    stack_item *optop;
    optop = ee->current_frame->optop;
    if (info->ret_size > 0) {
      printf("  optop[-1]: 0x%x %d %g\n",
		optop[-1].i, optop[-1].i, optop[-1].f);
      if (info->ret_size > 1) {
	printf("  optop[-2]: 0x%x %d %g\n",
		optop[-2].i, optop[-2].i, optop[-2].f);
      }
      fflush(stdout);
    }
  }
#endif

  return retsize;
}


/*
 * returns: method to be invoked, NULL if an exception occurred.
 */
struct methodblock *getInterfaceMethod(
#ifdef RUNTIME_DEBUG
	int runtime_debug,
#endif
	ExecEnv *ee,
	JHandle *obj, struct methodblock *imethod, unsigned char *guessptr) {
  struct methodblock *method;
  ClassClass *intfClazz;
  unsigned int offset;

  ClassClass *cb;
  struct methodtable *mtable;
  struct imethodtable *imtable;
  int guess;

  intfClazz = fieldclass(&imethod->fb);
  offset = imethod->fb.u.offset;
  guess = *guessptr;
#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    printf("  intf method: %s#%s %s, guess: %d\n",
	cbName(intfClazz), imethod->fb.name, imethod->fb.signature,
	guess);
    fflush(stdout);
  }
#endif

  if (!obj) {
    SignalError(ee, JAVAPKG "NullPointerException", 0);
    return NULL;
  }

  if (obj_flags(obj) == T_NORMAL_OBJECT) {
    mtable = obj_methodtable(obj);  cb = mtable->classdescriptor;
#ifdef METAVM
    if ((mtable == proxy_methodtable) && (GET_REMOTE_FLAG(ee))) {
      /* obj instanceof Proxy*/
      cb = unhand((HNET_shudo_metavm_Proxy *)obj)->clazz;
      mtable = cbMethodTable(cb);
    }
#endif	/* METAVM */
  }
  else {
    cb = classJavaLangObject;  mtable = cbMethodTable(cb);
  }
  imtable = cbIntfMethodTable(cb);

  /* `guess' is already calculated. */
  if (guess >= 0 && guess < imtable->icount &&
	imtable->itable[guess].classdescriptor == intfClazz) {
    goto getintf_done;
  }

  if (guess < 0 || guess >= imtable->icount ||
	imtable->itable[guess].classdescriptor != intfClazz) {
    for (guess = imtable->icount - 1; ;guess--) {
      if (guess < 0) {
	invokeInterfaceError(ee, NULL, cb, intfClazz);
	return NULL;
      }
      if (imtable->itable[guess].classdescriptor == intfClazz) {
	*guessptr = (unsigned char)guess;
	break;
      }
    }
  }

getintf_done:
  method = mt_slot(mtable, imtable->itable[guess].offsets[offset]);

#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    printf("  getInterfaceMethod() returns %s#%s\n",
		cbName(method->fb.clazz), method->fb.name);
    fflush(stdout);
  }
#endif

  return method;
}


JHandle *multianewarray(
#ifdef RUNTIME_DEBUG
	int runtime_debug,
#endif
	ExecEnv *ee, int dim, ClassClass *arrayclazz,
	stack_item *stackpointer) {
  stack_item *optop = ee->current_frame->optop;
  int i;

  stackpointer += (dim - 1);
  for (i = 0; i < dim; i++) {
    int size = stackpointer->i;
#ifdef RUNTIME_DEBUG
    if (runtime_debug) {
      printf("  %d: %d\n", i, size);
      fflush(stdout);
    }
#endif
#ifndef NO_CHECK
    if (size < 0)	/* NegativeArraySizeException */
      return (JHandle *)-1;
#endif
    optop[i].i = size;
    stackpointer--;
  }

  return MultiArrayAlloc(dim, arrayclazz, optop);
}


#ifndef NO_REWRITE
int once_in_new(ExecEnv *ee, ClassClass *cb) {
  unsigned short access;
  struct methodblock *mb;
  ClassClass *cur_cb;

  access = cbAccess(cb);
  if (access & (ACC_INTERFACE | ACC_ABSTRACT)) {
    SignalError(ee, JAVAPKG "InstantiationError", 0);
    return 1;
  }
#  ifndef NO_CHECK
  mb = ee->current_frame->current_method;
  cur_cb = (mb ? (mb->fb.clazz): NULL);
  if (!VerifyClassAccess(cur_cb, cb, TRUE)) {
    SignalError(ee, JAVAPKG "IllegalAccessError", 0);
    return 1;
  }
#  endif	/* NO_CHECK */

#  if !defined(INITCLASS_IN_COMPILING) && !defined(NO_REWRITE)
  if (!CB_INITIALIZED(cb))  InitClass(cb);
#  endif

  return 0;
}
#endif	/* !NO_REWRITE */


#if (JDK_VER < 12)
void InitClass(ClassClass *cb) {
  char *detail = 0;
#ifdef RUNTIME_DEBUG
  printf("InitClass(%s).\n", (cb?cbName(cb):"(null)"));
  fflush(stdout);
#endif
  ResolveClass(cb, &detail);
}
#endif	/* JDK_VER */


struct CatchFrame *searchCatchFrame(
	ExecEnv *ee, struct methodblock *mb, int bytepcoff
#ifdef RUNTIME_DEBUG
	, int runtime_debug
#endif
) {
  ClassClass *methodClazz;
  cp_item_type *constant_pool;
  unsigned char *type_table;
  struct CatchFrame *cf;
  int found;
  int i;

#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    printf("  searchCatchFrame() called:\n"
	   "    ee: 0x%x, mb: 0x%x, 0x%x\n",
	(int)ee, (int)mb, bytepcoff);
    fflush(stdout);
  }
#endif

  methodClazz = fieldclass(&mb->fb);
  constant_pool = cbConstantPool(methodClazz);
  type_table = constant_pool[CONSTANT_POOL_TYPE_TABLE_INDEX].type;

#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    JHandle *exc = ee->exception.exc;
    printf("  exc: ");
    if (exc)
      printf("0x%08x(%s)\n",
	(int)exc, cbName(exc->methods->classdescriptor));
    else
      printf(" is NULL !!!\n");
    printf("  bytepcoff %d\n", bytepcoff);
    fflush(stdout);
  }
#endif

  found = 0;	/* false */
  cf = mb->exception_table;
  for (i = 0; i < mb->exception_table_length; i++) {
#ifdef RUNTIME_DEBUG
    if (runtime_debug) {
      printf("  pc: start %ld end %ld type %d\n",
		cf->start_pc, cf->end_pc, cf->catchType);
      fflush(stdout);
    }
#endif
    if ((cf->start_pc <= bytepcoff) && (cf->end_pc > bytepcoff)) {
      int typeindex = cf->catchType;
      char *catchName;
      ClassClass *catchClazz = NULL;
      ClassClass *excClazz;

      if (typeindex == 0) {
		/* This catch frame accpets any type of exceptions. */
	found = 1;  goto search_finish;
      }

      catchName = GetClassConstantClassName(constant_pool, typeindex);
#ifdef RUNTIME_DEBUG
      if (runtime_debug) {
	printf("    catch name: %s\n", catchName);
	fflush(stdout);
      }
#endif

      for (excClazz = obj_array_classblock(ee->exception.exc);
		excClazz; excClazz = cbSuperclass(excClazz)) {
	if ( !(strcmp(cbName(excClazz), catchName)) &&
		(cbLoader(excClazz) == cbLoader(methodClazz)) ) {
	  found = 1;  goto search_finish;
	}

	if (!catchClazz) {
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, typeindex))
	    if (!(ResolveClassConstantFromClass2(methodClazz, typeindex, ee,
			1 << CONSTANT_Class, FALSE))) {
#ifdef RUNTIME_DEBUG
	      if (runtime_debug) {
		printf("    resolution of catch class failed: %d\n",
			typeindex);
		fflush(stdout);
	      }
#endif
	      continue;
	    }
	  catchClazz = constant_pool[typeindex].clazz;
	}
	if (excClazz == catchClazz) {
	  found = 1;  goto search_finish;
	}
      }
    }	/* if (start_pc <= pc < end_pc) */

    cf++;
  }	/* for (i = 0; ... */

search_finish:
  if (found) {
#ifdef RUNTIME_DEBUG
    if (runtime_debug) {
      printf("  exc. handler is found: %s#%s %ld-%ld type:%d\n"
		"  native offset 0x%x(%d).\n",
		cbName(mb->fb.clazz), mb->fb.name,
		cf->start_pc, cf->end_pc, (int)cf->catchType,
		(int)cf->compiled_CatchFrame, (int)cf->compiled_CatchFrame);
      fflush(stdout);
    }
#endif
    exceptionClear(ee);

    /* push exception object */
    ee->current_frame->optop = ee->current_frame->ostack;
    /* ee->current_frame->optop = ... */
    (ee->current_frame->optop++)->h = ee->exception.exc;

    return cf;
  }
  else {
#ifdef RUNTIME_DEBUG
    if (runtime_debug) {
      printf("  exc. handler is not found at method %s#%s.\n",
		cbName(mb->fb.clazz), mb->fb.name);
      fflush(stdout);
    }
#endif
    return NULL;
  }
}


void showStackFrames(ExecEnv *ee) {
  JavaFrame *frame = ee->current_frame;
  printf("stack frames:\n");
  while (frame) {
    struct methodblock *frameMb = frame->current_method;
    if (frameMb) {
      printf("  %s#%s %s 0x%x  ",
	cbName(frameMb->fb.clazz), frameMb->fb.name, frameMb->fb.signature,
	(int)frameMb);
      if (frame->lastpc)
	printf("pc: %d", frame->lastpc - frameMb->code);
      if (frameMb->invoker == sym_invokeJITCompiledMethod)
	printf(" (compiled)");
      printf("\n");
    }
    else {
      printf("  (null)  lastpc: 0x%08x\n", (int)frame->lastpc);
    }

    fflush(stdout);

    frame = frame->prev;
  }
}


#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
typedef struct Classjava_lang_Throwable {
  struct Hjava_lang_Object *backtrace;
  struct Hjava_lang_String *detailMessage;
} Classjava_lang_Throwable;
HandleTo(java_lang_Throwable);

void showExcStackTrace(JHandle *o) {
#if JDK_VER < 12
  HArrayOfByte *backtrace =
	(HArrayOfByte *)(unhand((Hjava_lang_Throwable *)o)->backtrace);
  if (backtrace) {
    unsigned char **data = (unsigned char **)(unhand(backtrace)->body);
    unsigned char **end = (unsigned char **)(data + obj_length(backtrace));
#else
  HArrayOfObject *backtrace =
	(HArrayOfObject *)(unhand((Hjava_lang_Throwable *)o)->backtrace);
  HArrayOfInt *hdata;
  if (backtrace)  hdata = (HArrayOfInt *)(unhand(backtrace)->body[0]);
  if (hdata) {
    unsigned char **data = (unsigned char **)(unhand(hdata)->body);
    unsigned char **end = (unsigned char **)(data + obj_length(hdata));
#endif	/* JDK_VER */
    for (; data < end; data++) {
      if (*data) {
#define BUF_SIZE	256
	char buf[BUF_SIZE];
	char *p = buf;
	struct methodblock *mb = methodByPC(*data);

	strncpy(buf, "\tat ", 4);  p += 4;
	if (mb) {
	  p += snprintf(p, buf + BUF_SIZE - p, "%s#%s %s  ",
		cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
	  p += snprintf(p, buf + BUF_SIZE - p, "pc: %d",
		*data - mb->code);
	  if (mb->invoker == sym_invokeJITCompiledMethod)
	    p += snprintf(p, buf + BUF_SIZE - p, "(compiled)");
	}
	else {
	  p += snprintf(p, buf + BUF_SIZE - p, "(null)");
	}
	printf(buf);
	printf("\n");
      }
    }
  }
}
#endif	/* RUNTIME_DEBUG || COMPILE_DEBUG */


#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG) || defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
struct methodblock *methodByPC(unsigned char *pc) {
  struct methodblock *mb;
  ClassClass *cb;
  int i, j;

#if JDK_VER < 12
  BINCLASS_LOCK();
#else
  BINCLASS_LOCK(sysThreadSelf());
#endif
  for (i = nbinclasses; --i >= 0; ) {
    cb = binclasses[i];
    for (mb = cbMethods(cb), j = cbMethodsCount(cb); --j >= 0; mb++) {
      if (mb->fb.access & ACC_NATIVE) {	/* native method */
	if (mb->code == pc)  goto done;
      }
      else {				/* not native method */
	if (mb->fb.access & ACC_MACHINE_COMPILED)	/* compiled method */
	  if (PCinCompiledCode(pc, mb))
	    goto done;
	else {
	  unsigned char *code = (unsigned char *)mb->code;
	  if (pc >= code && pc < code + mb->code_length)
	    goto done;
	}
      }
    }
  }
  mb = 0;

done:
#if JDK_VER < 12
  BINCLASS_UNLOCK();
#else
  BINCLASS_UNLOCK(sysThreadSelf());
#endif
  return mb;
}
#endif	/* RUNTIME_DEBUG || COMPILE_DEBUG || EXC_SIGNAL || GET_SIGCONTEXT */


#ifdef RUNTIME_DEBUG
char *showObjectBody(char *sig, JHandle *obj) {
  switch (*sig) {
  case '[':
    sig++;
    if (*sig == 'L')  while (*(sig++) != ';');
    else sig++;
#ifdef METAVM
    if (obj)
      if (obj_flags(obj) == T_NORMAL_OBJECT)
	goto show_obj;
#endif	/* METAVM */
    if (!obj)
      printf("(null)");
    else {
      char buf[256];
show_array:
      printf("len %ld", obj_length(obj));
      switch (obj_flags(obj)) {
      case T_BYTE:
	printf(" [B `%s'", unhand((HArrayOfByte *)obj)->body);
	break;
      case T_CHAR:
	printf(" [C `%s'",
		unicode2utf(unhand((HArrayOfChar *)obj)->body, obj_length(obj),
			 buf, 256));
	break;
      }
      printf(" flags %x", obj_flags(obj));
    }
    break;
  case 'L':
    while (*(sig++) != ';');
show_obj:
    if (!obj)
      printf("(null)");
    else {
      ClassClass *cb;
      char *classname;
      if (obj_flags(obj) != T_NORMAL_OBJECT)  goto show_array;
      if (obj->methods) {
	classname = cbName(obj_array_classblock(obj));
	printf("class %s", classname);
	if (!strcmp(classname, "java/lang/String")) {
	  int strlen = javaStringLength((Hjava_lang_String *)obj);
	  printf(" len %d", strlen);
	  if (strlen) {
	    char buf[256];
	    printf(" `");
	    javaString2CString((Hjava_lang_String *)obj, buf, 256);
	    printf(buf);
	    printf("'");
	  }
	}
	else if (!strcmp(classname, "java/lang/Class")) {
	  printf(" (%s)", cbName((ClassClass *)obj));
	}
      }	/* obj->methods */
    }
    break;
  default:
    sig++;
  }

  return sig;
}
#endif	/* RUNTIME_DEBUG */
