/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

#include <string.h>	/* for strcmp() */

#include "compiler.h"

#if JDK_VER >= 12
extern sys_mon_t *monitorEnter2(ExecEnv *, uintptr_t);
extern int monitorExit2(ExecEnv *, uintptr_t);
#endif	/* JDK_VER */

#if 0
#  include "threads.h"	/* for JavaStackSize */
#else
extern long JavaStackSize;
#endif


/*
 * Compile and invoke the method.
 */
bool_t compileAndInvokeMethod(
	JHandle *o, struct methodblock *mb, int args_size, ExecEnv *ee) {
  int access;
  int compilation_result;
  MONITOR_T key;
  int invocation_count;

  key = obj_monitor(fieldclass(&mb->fb));

#ifdef COMPILE_DEBUG
  printf("\n");
  printf("compileAndInvokeMethod() called by %x:\n  %s#%s %s\n",
	(int)sysThreadSelf(),
	cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature);
  fflush(stdout);
#if 1
  if (debugp(mb)) {
    showStackFrames(ee);
  }
#else
  debugp(mb);
#endif
#endif

  access = mb->fb.access;

  /* compile the called method */
  monitorEnter2(ee, key);
  {
    bool_t (*invoker)(JHandle*,struct methodblock *,int,ExecEnv*);

    /* add a counter */
    invocation_count = ((CodeInfo *)mb->CompiledCodeInfo)->invocation_count;
    invocation_count++;
    ((CodeInfo *)mb->CompiledCodeInfo)->invocation_count = invocation_count;
#ifdef COMPILE_DEBUG
    printf("  invocation count: %d\n", invocation_count);  fflush(stdout);
#endif

    invoker = mb->invoker;
    if (invoker == sym_invokeJITCompiledMethod) {
#ifdef COMPILE_DEBUG
      printf("  already compiled.\n");  fflush(stdout);
#endif
      monitorExit2(ee, key);
      goto compilation_done;
    }
    else if ((invoker != sym_compileAndInvokeMethod)	/* being compiled */
		|| (invocation_count < opt_cmplthreshold)
#ifndef IGNORE_DISABLE
		|| (!compiler_enabled)		/* compiler is disabled */
#endif	/* IGNORE_DISABLE */
		) {
      monitorExit2(ee, key);
      goto candi_call_normal_method;
    }
  }

  /* compile */
#ifdef REENTER_DEBUG
  mb->invoker = redirectInvoker;
#else
  mb->invoker = access2invoker(access);
	/* the method being compiled is to be interpreted */
#endif
#ifdef COMPILE_DEBUG
  printf("  now compile.\n");  fflush(stdout);
#endif
  monitorExit2(ee, key);

  {
    JavaFrame *frame = ee->current_frame;
    frame->optop += args_size;	/* to save real args */
	/* not to be broken args by method invocation during compilation */
    compilation_result = compileMethod(mb);
	/* invoker is set to invokeJITCompiledMethod() */
    frame->optop -= args_size;	/* restore real optop */
  }

  if (compilation_result) {
      /* fail to compile... */
#ifdef COMPILE_DEBUG
    printf("  compilation failed...\n");  fflush(stdout);
#endif
    mb->invoker = access2invoker(access);
    goto candi_call_normal_method;
  }

compilation_done:

#ifdef COMPILE_DEBUG
  printf("compileAndInvokeMethod() now call invoker (%s):\n  %s#%s %s\n",
	nameOfInvoker(mb->invoker),
	cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature);
  fflush(stdout);
#endif
  return mb->invoker(o, mb, args_size, ee);

candi_call_normal_method:
  {
    bool_t (*norm_invoker)(JHandle*,struct methodblock *,int,ExecEnv*);
    bool_t invoker_ret;

    norm_invoker = access2invoker(access);
#ifdef COMPILE_DEBUG
    printf("call original invoker: %s\n",
		nameOfInvoker(norm_invoker));
    fflush(stdout);
#endif
    invoker_ret = norm_invoker(o, mb, args_size, ee);
    if (!invoker_ret)  return FALSE;

    if (!(access & (ACC_ABSTRACT | ACC_NATIVE))) {
	/* normal Java method */

      /* restack is done in invocationHelper() (in runtime.c)
	 if this method is called by compiled method */

      int exec_ret;
      stack_item *optop, *old_optop;
      int retsize;
      JavaFrame *cur_frame;;

#ifdef RUNTIME_DEBUG
      printf("call ExecuteJava(invoker.c): %s#%s.\n",
	cbName(mb->fb.clazz), mb->fb.name);  fflush(stdout);
#endif
      exec_ret = ExecuteJava(mb->code, ee);
#ifdef RUNTIME_DEBUG
      printf("ExecuteJava(invoker.c) done: %s#%s.\n",
	     cbName(mb->fb.clazz), mb->fb.name);  fflush(stdout);
#endif
      cur_frame = ee->current_frame;
      old_optop = cur_frame->optop;
#if defined(EXECUTEJAVA_IN_ASM) && (JDK_VER < 12)
      if (cur_frame->monitor) {
	monitorExit2(ee, (MONITOR_T)cur_frame->monitor);
      }
#endif	/* EXECUTEJAVA_IN_ASM */
      cur_frame = cur_frame->prev;
      ee->current_frame = cur_frame;
      if (!exec_ret) {
#ifdef RUNTIME_DEBUG
	printf("  ExecuteJava() returned false.\n");
	if (exceptionOccurred(ee))
	  printf("  clazz of exc: %s\n",
		cbName(ee->exception.exc->methods->classdescriptor));
	fflush(stdout);
#endif
	return FALSE;
      }

#ifdef EXECUTEJAVA_IN_ASM
      /* This operation is required
	 only with x86 assembly ver. of executeJava.c */
      retsize = ((CodeInfo *)mb->CompiledCodeInfo)->ret_size;
#  ifdef RUNTIME_DEBUG
      printf("  retsize: %d\n", retsize);  fflush(stdout);
#  endif
      if (retsize != 0) {
	optop = cur_frame->optop;
	if (retsize == 1) {
	  optop[0] = old_optop[-1];
#  ifdef RUNTIME_DEBUG
	  printf("  optop[0]: %08x\n", optop[0].i);  fflush(stdout);
#  endif
	  optop++;
	}
	else {	/* retsize == 2 */
	  optop[0] = old_optop[-2];  optop[1] = old_optop[-1];
#  ifdef RUNTIME_DEBUG
	  printf("  optop[0,1]: %08x,%08x\n", optop[0].i, optop[1].i);
	  fflush(stdout);
#  endif
	  optop += 2;
	}
	cur_frame->optop = optop;
      }	/* if (retsize != 0) */
#endif	/* EXECUTEJAVA_IN_ASM */
    }	/* normal Java method */

    return invoker_ret;
  }
}


#ifdef REENTER_DEBUG
bool_t redirectInvoker(
	JHandle *o, struct methodblock *mb, int args_size, ExecEnv *ee) {
  bool_t (*invoker)(JHandle*,struct methodblock *,int,ExecEnv*);
  bool_t invoker_ret;

#ifdef RUNTIME_DEBUG
  printf("\nredirectInvoker() called:\n  %s#%s %s\n",
	cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature);
  fflush(stdout);
#endif

  invoker = access2invoker(mb->fb.access);
  invoker_ret = invoker(o, mb, args_size, ee);

#ifdef RUNTIME_DEBUG
  if (exceptionOccurred(ee)) {
    JHandle *exc = ee->exception.exc;
    printf("exception occurred:\n  %s\n",
	cbName(exc->methods->classdescriptor));
    fflush(stdout);
  }
#endif

  return invoker_ret;
}
#endif	/* REENTER_DEBUG */


/*
 * Invoke the compiled method.
 */
bool_t invokeJITCompiledMethod(
	JHandle *o, struct methodblock *mb, int args_size, ExecEnv *ee) {
  JavaFrame *old_frame, *frame;
  struct methodblock *prev_method;	/* for restack */
  int access;
  stack_item *native_vars;
#ifdef RUNTIME_DEBUG
  int runtime_debug;
#endif

  /* allocate local var. space */
	/* caution: This code segment must be placed at head of function. */
  asm("leal  -4(%%esp),%%edx\n\t"
      "subl  %1,%%esp\n\t"
      "movl  %%edx,%0"
	: "=m" (native_vars) : "r" (mb->nlocals * 4)
	: "edx");
	/* native_vars = %esp - 4 */
	/* %esp -= mb->nlocals * 4 */

#ifdef RUNTIME_DEBUG
  runtime_debug = debugp(mb);
#endif	/* RUNTIME_DEBUG */

#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    printf("\n");
    printf("invokeJITCompiledMethod() called by %x:\n  %s#%s %s 0x%x\n",
	(int)sysThreadSelf(),
	cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature, (int)mb);
    if (exceptionOccurred(ee)) {
      printf("  An exception not handled remains!\n");
    }
    fflush(stdout);
#if 1
    showStackFrames(ee);
#endif
  }	/* if (runtime_debug) */
#endif	/* RUNTIME_DEBUG */

  /* add a counter */
  ((CodeInfo *)mb->CompiledCodeInfo)->invocation_count++;

  old_frame = ee->current_frame;
  access = mb->fb.access;

  /* creates a new frame */
#if 1
  {
    JavaStack *stack = old_frame->javastack;
    stack_item *optop = old_frame->optop;

    frame = (JavaFrame *)(optop + mb->nlocals);
    if (frame->ostack + mb->maxstack >= stack->end_data)
    {
#ifdef RUNTIME_DEBUG
      if (runtime_debug) {
	printf("stack chunk overflow, provide next chunk.\n");
	fflush(stdout);
      }
#endif	/* RUNTIME_DEBUG */
#  if JDK_VER < 12
      if (stack->next)  stack = stack->next;
      else {
	if (stack->stack_so_far + JAVASTACK_CHUNK_SIZE * sizeof(stack_item)
		> JavaStackSize) {
	  SignalError(ee, JAVAPKG "StackOverflowError", 0);
	  return FALSE;
	}
	if (!(stack = CreateNewJavaStack(ee, stack))) {
	  SignalError(ee, JAVAPKG "OutOfMemoryError", 0);
	  return FALSE;
	}
      }
      frame = (JavaFrame *)(stack->data + mb->nlocals);
	/* needless to copy args to a frame on new stack chunk */
#  else
      {
	JavaStack *tmp_stack = stack;
	JavaFrame *tmp_frame = frame;
	stack_item *tmp_optop = optop;
	if (!ExpandJavaStack(ee, &tmp_stack, &tmp_frame, &tmp_optop,
			args_size, mb->nlocals, mb->maxstack))
	  return FALSE;
	stack = tmp_stack;
	frame = tmp_frame;
	optop = tmp_optop;
      }
#  endif	/* JDK_VER */
    }
#if 1
    frame->constant_pool = NULL;
#else
    frame->constant_pool = cbConstantPool(fieldclass(&mb->fb));
#endif
    frame->returnpc = frame->lastpc = mb->code;
	/* lastpc is not initialized in invoke*JavaMethod() */
    frame->optop = frame->ostack;
    frame->vars = optop;
    frame->prev = old_frame;
    frame->javastack = stack;
    frame->current_method = mb;

    ee->current_frame = frame;
  }	/* create a new frame */
#else
  invokeJavaMethod(o, mb, args_size, ee);

  frame = ee->current_frame;
  frame->lastpc = mb->code;
	/* lastpc is not initialized in invoke*JavaMethod() */
#endif

  /* call */

  prev_method = old_frame->current_method;
  if (prev_method) {
    unsigned char *prev_returnpc = old_frame->returnpc;
    if (prev_returnpc &&
	((prev_returnpc < prev_method->code) ||
	 ((prev_method->code + prev_method->code_length) <= prev_returnpc))) {
      /* called by compiled method */
      frame->vars = (stack_item *)prev_returnpc;
      goto restack2native_done;
    }
  }

  /* called by normal Java or native method */
  /* restack from JVM stack to native stack */
  {
    char *argsizes;
    stack_item *args;

    argsizes = ((CodeInfo *)mb->CompiledCodeInfo)->argsizes;
    args = frame->vars;	/* points to JVM stack */

    frame->vars = native_vars;

    while (*argsizes) {
      if (*argsizes == 1) {
	native_vars[0] = args[0];
	native_vars--;  args++;
      }
      else {
	native_vars[-1] = args[0];
	native_vars[0] = args[1];
	native_vars -= 2;  args += 2;
      }
      argsizes++;
    }	/* while (*argsizes) */
  }	/* restack */

restack2native_done:

#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    CodeInfo *info = (CodeInfo *)mb->CompiledCodeInfo;

    printf("  nlocals: %d\n", mb->nlocals);
    printf("  o:0x%08x, mb:0x%08x, args_size:%d, ee:0x%08x\n",
	(int)o, (int)mb, args_size, (int)ee);
    printf("  mb->CompiledCode: 0x%08x\n", (int)mb->CompiledCode);
    printf("  code size: 0x%08x\n", (info ? info->code_size : 0));
    fflush(stdout);

    {
      stack_item *vars = ee->current_frame->vars;
      char *sig;
      int i;

      if (args_size > 0) {
	printf("  args:\n");
	i = 0;
	if (!(access & ACC_STATIC)) {
	  printf("    0x%08x @ 0x%08x  ", (int)vars[0].h, (int)vars);
	  showObjectBody("L;", vars[0].h);
	  printf("\n");
	  i++;
	}
	sig = mb->fb.signature;
	if (sig[0] == '(')  sig++;

	for (; i < args_size; i++) {
	  JHandle *arg = vars[-i].h;
	  if ((sig[0] == 'D') || (sig[0] == 'J'))  i++;
	  printf("    0x%08x @ 0x%08x  ", (int)arg, (int)(vars - i));
	  sig = showObjectBody(sig, arg);
	  printf("\n");
	}
      }
      fflush(stdout);
    }
  }
#endif


  /* monitor */
  if (!(access & ACC_SYNCHRONIZED))
    frame->monitor = NULL;
  else {
    frame->monitor =
#if JDK_VER >= 12
      (struct sys_mon *)
#endif
      o;
    monitorEnter2(ee, obj_monitor(o));
  }

#ifdef ENABLE_PROFILING
  /* profiling */
  if (java_monitor)  frame->mon_starttime = now();
#endif

#ifdef RUNTIME_DEBUG
  ((void (*) (JHandle *, struct methodblock *, int, ExecEnv *, int))
	mb->CompiledCode)(o, mb, args_size, ee, runtime_debug);
#else
  ((void (*) (JHandle *, struct methodblock *, int, ExecEnv *))
	mb->CompiledCode)(o, mb, args_size, ee);
#endif

  if (access & ACC_SYNCHRONIZED) {	/* if (frame->monitor) */
    monitorExit2(ee, (MONITOR_T)frame->monitor);
  }

#ifdef ENABLE_PROFILING
  if (java_monitor)
    java_mon(prev_method, mb, now() - frame->mon_starttime);
#endif

  ee->current_frame = frame->prev;

#ifdef RUNTIME_DEBUG
  if (runtime_debug) {
    printf("compiled method finished by %x: %s#%s %s\n",
	(int)sysThreadSelf(),
	cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature);
    fflush(stdout);

    printf("  free memory: handle %lld obj %lld\n",
		FreeHandleMemory(), FreeObjectMemory());
    fflush(stdout);

    if (exceptionOccurred(ee)) {
      JHandle *exc = ee->exception.exc;
      printf("  exception occurred:\n");
      printf("    0x%08x(%s)\n",
		(int)exc, cbName(exc->methods->classdescriptor));
      showExcStackTrace(exc);
      fflush(stdout);
    }
    printf("\n");
    fflush(stdout);
  }	/* if (runtime_debug) */
#endif


#ifdef RUNTIME_DEBUG
  runtime_debug = 0;
#endif

  return !(exceptionOccurred(ee));
}


/*
 * Local functions
 */

#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
int debugp(struct methodblock *mb) {
  int runtime_debug = 0;

#  if 0
  runtime_debug = 1;
#  else
  if ((!strcmp(cbName(mb->fb.clazz), "javax/swing/UIManager"))
	&& (!strcmp(mb->fb.name, "maybeInitialize")))
    runtime_debug = 1;
  else if ((!strcmp(cbName(mb->fb.clazz), "sun/tools/java/Constants"))
	&& (!strcmp(mb->fb.name, "<clinit>")))
    runtime_debug = 1;

#if 0
  else if ((!strcmp(cbName(mb->fb.clazz), "sun/applet/AppletViewer"))
	&& (!strcmp(mb->fb.name, "<init>")))
    runtime_debug = 1;
  else if ((!strcmp(cbName(mb->fb.clazz), "sun/applet/AppletPanel"))
	&& (!strcmp(mb->fb.name, "sendEvent")))
    runtime_debug = 1;
  else if ((!strcmp(cbName(mb->fb.clazz), "sun/applet/AppletPanel"))
	&& (!strcmp(mb->fb.name, "getNextEvent")))
    runtime_debug = 1;
  else if ((!strcmp(cbName(mb->fb.clazz), "sun/awt/motif/MFramePeer"))
	&& (!strcmp(mb->fb.name, "<init>")))
    runtime_debug = 1;
  else if ((!strcmp(cbName(mb->fb.clazz), "sun/awt/motif/MToolkit"))
	&& (!strcmp(mb->fb.name, "createFrame")))
    runtime_debug = 1;
  else if ((!strcmp(cbName(mb->fb.clazz), "java/awt/Frame"))
	&& (!strcmp(mb->fb.name, "addNotify")))
    runtime_debug = 1;
  else if ((!strcmp(cbName(mb->fb.clazz), "java/awt/Window"))
	&& (!strcmp(mb->fb.name, "pack")))
    runtime_debug = 1;

  else if ((!strcmp(cbName(mb->fb.clazz), "java/lang/Thread"))
	&& (!strcmp(mb->fb.name, "<init>")))
    runtime_debug = 1;
  else if ((!strcmp(cbName(mb->fb.clazz), "java/lang/Thread"))
	&& (!strcmp(mb->fb.name, "init")))
    runtime_debug = 1;
  else if ((!strcmp(cbName(mb->fb.clazz), "java/lang/Thread"))
	&& (!strcmp(mb->fb.name, "exit")))
    runtime_debug = 1;

#endif

  else if (!strcmp(mb->fb.name, "main"))
    runtime_debug = 1;
  else if (!strcmp(mb->fb.name, "start"))
    runtime_debug = 1;
  else if (!strcmp(mb->fb.name, "run"))
    runtime_debug = 1;
#endif


  return runtime_debug;
}
#endif	/* RUNTIME_DEBUG */
