/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

#include <stdio.h>
#include <stdlib.h>	/* for malloc(), realloc() */
#include <string.h>	/* for memcpy() */

#include "compiler.h"


static CompilerContext *cc_pool = NULL;

static CompilerContext *newCompilerContext(struct methodblock *mb);
static void freeCompilerContext(CompilerContext *cc);
static void resetCompilerContext(CompilerContext *cc, struct methodblock *mb);


void *access2invoker(int access) {
  if (access & ACC_ABSTRACT) {
    return (void *)sym_invokeAbstractMethod;
  }
  else if (access & ACC_NATIVE) {
    return (void *)sym_invokeLazyNativeMethod;
  }
  else {
    if (access & ACC_SYNCHRONIZED) {
      return (void *)sym_invokeSynchronizedJavaMethod;
    }
    else {
      return (void *)sym_invokeJavaMethod;
    }
  }
}


char *nameOfInvoker(void *inv) {
  char *ret;

#define IF_A_INVOKER(NAME)	if (inv == sym_##NAME)  ret = #NAME
#define ELIF_A_INVOKER(NAME)	else IF_A_INVOKER(NAME)

  IF_A_INVOKER(compileAndInvokeMethod);
  ELIF_A_INVOKER(invokeJITCompiledMethod);
  ELIF_A_INVOKER(invokeJavaMethod);
  ELIF_A_INVOKER(invokeSynchronizedJavaMethod);
  ELIF_A_INVOKER(invokeAbstractMethod);
  ELIF_A_INVOKER(invokeNativeMethod);
  ELIF_A_INVOKER(invokeSynchronizedNativeMethod);
  ELIF_A_INVOKER(invokeJNINativeMethod);
  ELIF_A_INVOKER(invokeJNISynchronizedNativeMethod);
  ELIF_A_INVOKER(invokeLazyNativeMethod);
  else  ret = "<other>";

  return ret;
}


void showCompilerContext(CompilerContext *cc) {
  printf("compiler context: 0x%x\n", (int)cc);
  fflush(stdout);

  if (cc) {
    struct methodblock *mb = cc->mb;

    printf("  ExecEnv: 0x%08x\n", (int)cc->ee);
    printf("  method: %s#%s %s (0x%08x)\n",
	(mb ? cbName(mb->fb.clazz) : "(null)"),
	(mb ? mb->fb.name : "(null)"),
	(mb ? mb->fb.signature : "(null)"), (int)mb);
    printf("  buffer: 0x%08x\n", (int)cc->buffer);
    printf("  bufp  : 0x%08x offset:0x%x(%d)\n",
	(int)cc->bufp, cc->bufp - cc->buffer, cc->bufp - cc->buffer);
    printf("  pctable: 0x%08x\n", cc->pctable);
    printf("  jptable: 0x%08x\n", cc->jptable);
  }
  fflush(stdout);
}


CompilerContext *getCompilerContext(struct methodblock *mb) {
#if JDK_VER >= 12
  ExecEnv *ee = EE();	/* for monitor{Enter,Exit}2() */
#endif
  CompilerContext *ret;
  MONITOR_T key = obj_monitor((JHandle *)classJavaLangThreadDeath);

  monitorEnter2(ee, key);
  if (cc_pool) {
    ret = cc_pool;
    cc_pool = cc_pool->next;

    resetCompilerContext(ret, mb);
  }
  else {
    ret = newCompilerContext(mb);
  }
  monitorExit2(ee, key);

  return ret;
}


void releaseCompilerContext(CompilerContext *cc) {
#if JDK_VER >= 12
  ExecEnv *ee = EE();	/* for monitor{Enter,Exit}2() */
#endif

  MONITOR_T key = obj_monitor((JHandle *)classJavaLangThreadDeath);

  monitorEnter2(ee, key);
  cc->next = cc_pool;
  cc_pool = cc;
  monitorExit2(ee, key);
}


static CompilerContext *newCompilerContext(struct methodblock *mb) {
  CompilerContext *cc;

#ifdef COMPILE_DEBUG
  printf("newCompilerContext(): mb is 0x%08x(%s)\n",
		(int)mb, (mb ? mb->fb.name : "(null)"));
  fflush(stdout);
#endif
  cc = (CompilerContext *)sysMalloc(sizeof(CompilerContext));
  if (!cc) {
#ifdef COMPILE_DEBUG
    printf("newCompilerContext(): cannot allocate memory.\n");
    fflush(stdout);
#endif
    return NULL;
  }

  cc->buffer = (unsigned char *)sysMalloc(DEFAULT_BUF_SIZE);
  cc->buf_size = DEFAULT_BUF_SIZE;

  cc->pctablesize = DEFAULT_PCTABLE_SIZE;
  cc->pctable = (pcentry *)sysMalloc(sizeof(pcentry) * DEFAULT_PCTABLE_SIZE);

  cc->jptablesize = DEFAULT_JPTABLE_SIZE;
  cc->jptable = (jpentry *)sysMalloc(sizeof(jpentry) * DEFAULT_JPTABLE_SIZE);

  resetCompilerContext(cc, mb);

  return cc;
}


static void freeCompilerContext(CompilerContext *cc) {
#ifdef COMPILE_DEBUG
  printf("freeCompilerContext(): cc is 0x%08x\n", (int)cc);
  fflush(stdout);
#endif
  if (cc) {
#ifdef COMPILE_DEBUG
    printf("  buffer: 0x%08x\n  pctable: 0x%08x\n  jptable: 0x%08x\n",
	(int)cc->buffer, (int)cc->pctable, (int)cc->jptable);
    fflush(stdout);
#endif
    if (cc->buffer)  sysFree(cc->buffer);
    if (cc->pctable)  sysFree(cc->pctable);
    if (cc->jptable)  sysFree(cc->jptable);
    sysFree(cc);
#ifdef COMPILE_DEBUG
    fflush(stdout);
    fflush(stderr);
#endif
  }
#ifdef COMPILE_DEBUG
  printf("freeCompilerContext() done.\n");
  fflush(stdout);
#endif
}


static void resetCompilerContext(CompilerContext *cc, struct methodblock *mb) {
  cc->ee = EE();

  cc->mb = mb;

  cc->bufp = cc->buffer;

  pctableClear(cc);
  cc->jptablelen = 0;

  cc->next = NULL;
}


void writeToBuffer(CompilerContext *cc, void *ptr, size_t len) {
#ifdef BUF_DEBUG
  printf("writeToBuffer(cc, 0x%x, %d) called.\n", (int)ptr, (int)len);
  printf("  bufp 0x%x(offset:%d)\n", (int)cc->bufp, cc->bufp - cc->buffer);
  fflush(stdout);
#endif
  ensureBufferSize(cc, len);
  memcpy(cc->bufp, ptr, len);

  cc->bufp += len;
#ifdef BUF_DEBUG
  printf("writeToBuffer() done.\n");
  fflush(stdout);
#endif
}


void cancelOnBuffer(CompilerContext *cc, size_t len) {
#ifdef BUF_DEBUG
  printf("cancelOnBuffer(cc, %d) called.\n", (int)len);
  fflush(stdout);
#endif

  cc->bufp -= len;
}


void ensureBufferSize(CompilerContext *cc, size_t req) {
  int buf_offset = cc->bufp - cc->buffer;

  if (req > (cc->buf_size - buf_offset)) {

#ifdef BUF_DEBUG
    printf("ensureBufferSize(): now extend the space.\n");
    printf("  buffer 0x%x, bufp 0x%x, offset 0x%x(%d)\n",
		(int)cc->buffer, (int)cc->bufp, buf_offset, buf_offset);
    fflush(stdout);
#endif
#ifdef BUF_DEBUG
    printf("  buf_size: %d -> ", cc->buf_size);
#endif
    do {
      cc->buf_size <<= 1;
    } while (req > (cc->buf_size - buf_offset));
#ifdef BUF_DEBUG
    printf("%d.\n", cc->buf_size);
    fflush(stdout);
#endif

    cc->buffer = (unsigned char *)sysRealloc(cc->buffer, cc->buf_size);
    cc->bufp = cc->buffer + buf_offset;
  }
}


/* PC table */

#ifdef CODE_DB
void pctableExtend(CompilerContext *cc, uint32_t size) {
  if (size < cc->pctablesize)  return;

  cc->pctable = (pcentry *)sysRealloc(cc->pctable, sizeof(pcentry) * size);
  cc->pctablesize = size;
}
#endif	/* CODE_DB */


void pctableClear(CompilerContext *cc) {
  cc->pctablelen = 0;
}


void pctableAdd(CompilerContext *cc,
	int opcode, int state, unsigned int byteoff, unsigned int nativeoff) {
  pcentry *entryp;

  if (cc->pctablelen >= cc->pctablesize) {	/* extend table size */
    do
      cc->pctablesize <<= 1;
    while (cc->pctablelen >= cc->pctablesize);
    cc->pctable = (pcentry *)sysRealloc(cc->pctable,
				sizeof(pcentry) * cc->pctablesize);
  }

  entryp = cc->pctable + cc->pctablelen;
  entryp->opcode = opcode;
  entryp->state = state;
  entryp->byteoff = byteoff;
  entryp->nativeoff = nativeoff;

  (cc->pctablelen)++;
}


void pctableInsert(CompilerContext *cc, int index,
	int opcode, int state, unsigned int byteoff, unsigned int nativeoff) {
  pcentry *entryp;

  if (cc->pctablelen >= cc->pctablesize) {	/* extend table size */
    do { cc->pctablesize <<= 1; } while (cc->pctablelen >= cc->pctablesize);
    cc->pctable = (pcentry *)sysRealloc(cc->pctable,
				sizeof(pcentry) * cc->pctablesize);
  }

  entryp = cc->pctable + index;
  memmove(entryp + 1, entryp, sizeof(pcentry) * (cc->pctablelen - index));

  entryp->opcode = opcode;
  entryp->state = state;
  entryp->byteoff = byteoff;
  entryp->nativeoff = nativeoff;

  (cc->pctablelen)++;
}


void pctableDelete(CompilerContext *cc, int index) {
  pcentry *entryp;

  (cc->pctablelen)--;

  entryp = cc->pctable + index;
  memmove(entryp, entryp + 1, sizeof(pcentry) * (cc->pctablelen - index));
}


pcentry *pctableGet(CompilerContext *cc, int byteoff) {
  int l = 0;
  int h = cc->pctablelen;
  int index;
  pcentry *entryp;

  while (1) {
    index = l + ((h - l) >> 1);

    entryp = cc->pctable + index;
    if (entryp->byteoff == byteoff) {	/* found */
      while (--index >= 0) {
	if ((cc->pctable + index)->byteoff != byteoff)
	  break;
      }
      index++;
      return cc->pctable + index;
    }

    if (l == h)  return NULL;	/* not found */

    if (entryp->byteoff < byteoff)
      l = index + 1;
    else
      h = index;
  }
}


/* Jump instruction table */

void jptableAdd(CompilerContext *cc,
	unsigned int tgtoff, unsigned int argoff) {
  jpentry *entryp;

  if (cc->jptablelen >= cc->jptablesize) {	/* extend table size */
#ifdef COMPILE_DEBUG
    printf("jptableAdd(): extending jump table.\n");  fflush(stdout);
#endif
    do {
      cc->jptablesize <<= 1;
    } while (cc->jptablelen >= cc->jptablesize);

    cc->jptable = (jpentry *)sysRealloc(cc->jptable,
				sizeof(jpentry) * cc->jptablesize);
#ifdef COMPILE_DEBUG
    printf("jptableAdd() extending done: 0x%x\n", (int)(cc->jptable));
    fflush(stdout);
#endif
  }

  entryp = cc->jptable + cc->jptablelen;
  entryp->tgtoff = tgtoff;
  entryp->argoff = argoff;

  (cc->jptablelen)++;
}


CodeInfo *prepareCompiledCodeInfo(ExecEnv *ee, struct methodblock *method) {
  CodeInfo *info = (CodeInfo *)method->CompiledCodeInfo;
  if (!info) {
    info = (CodeInfo *)sysMalloc(sizeof(CodeInfo));
    memset(info, 0, sizeof(CodeInfo));	/* fill zero into CodeInfo */
    method->CompiledCodeInfo = (void *)info;
  }
  if (!info->argsizes) {
#define AS_CHUNK_SIZE	256
    char *sig = method->fb.signature + 1;
    char sizearr[AS_CHUNK_SIZE];
    int index = 0, len = 0;

    if (!(method->fb.access & ACC_STATIC))
      sizearr[index++] = 1;	/* receiver */

    while (*sig != ')') {
      if ((*sig == 'D') || (*sig == 'J')) {	/* long or double */
	sizearr[index++] = 2;
	sig++;
      }
      else {
	sizearr[index++] = 1;
	if (*sig == 'L')  while (*(sig++) != ';');
	else if (*sig == '[') {
	  do { sig++; } while (*sig == '[');
	  if (*sig == 'L')  while (*(sig++) != ';');
	  else  sig++;
	}
	else
	  sig++;
      }

      if (index >= AS_CHUNK_SIZE) {
	if (info->argsizes)
	  info->argsizes = sysRealloc(info->argsizes, len + AS_CHUNK_SIZE);
	else
	  info->argsizes = sysMalloc(len + AS_CHUNK_SIZE);
	memcpy(info->argsizes + len, sizearr, AS_CHUNK_SIZE);
	len += AS_CHUNK_SIZE;
	index = 0;
      }
    }	/* while (*sig != ')') */

    sizearr[index++] = 0;	/* terminate */

    if (info->argsizes)
      info->argsizes = sysRealloc(info->argsizes, len + index);
    else
      info->argsizes = sysMalloc(len + index);
    memcpy(info->argsizes + len, sizearr, index);

    sig++;
    if (*sig == 'V')
      info->ret_size = 0;
    else if ((*sig == 'D') || (*sig == 'J'))
      info->ret_size = 2;
    else
      info->ret_size = 1;
  }

#ifdef EXC_BY_SIGNAL
  info->throwtablesize = DEFAULT_THROWTABLE_SIZE;
  info->throwtablelen = 0;
  info->throwtable =
	(throwentry *)sysMalloc(sizeof(throwentry) * DEFAULT_THROWTABLE_SIZE);
#endif	/* EXC_BY_SIGNAL */

  return info;
}


void freeCompiledCodeInfo(CodeInfo *info) {
  if (info) {
#ifdef EXC_BY_SIGNAL
    if (info->throwtable)  sysFree(info->throwtable);
#endif	/* EXC_BY_SIGNAL */
    if (info->argsizes)  sysFree(info->argsizes);
    sysFree(info);
  }
}


#ifdef EXC_BY_SIGNAL
#ifdef CODE_DB
void throwtableExtend(CodeInfo *info, uint32_t size) {
  if (size < info->throwtablesize)  return;

  info->throwtable =
	(throwentry *)sysRealloc(info->throwtable, sizeof(throwentry) * size);
  info->throwtablesize = size;
}
#endif	/* CODE_DB */


void throwtableAdd(CodeInfo *info, uint32_t start, uint16_t len,
	unsigned char byteoff, unsigned char opcode) {
  throwentry *entryp;

  if (info->throwtablelen >= info->throwtablesize) {	/* extend table size */
    do
      info->throwtablesize <<= 1;
    while (info->throwtablelen >= info->throwtablesize);
    info->throwtable = (throwentry *)sysRealloc(info->throwtable,
				sizeof(throwentry) * info->throwtablesize);
  }

  entryp = info->throwtable + info->throwtablelen;
  entryp->start = start;
  entryp->len = len;
  entryp->byteoff = byteoff;
  entryp->opcode = opcode;

  (info->throwtablelen)++;
}


throwentry *throwtableGet(CodeInfo *info, uint32_t nativeoff) {
  int l = 0;
  int h = info->throwtablelen;
  int index;
  throwentry *entryp;

  while (1) {
    index = l + ((h - l) >> 1);

    entryp = info->throwtable + index;
    if ((entryp->start <= nativeoff) &&
	(nativeoff < (entryp->start + entryp->len))) {	/* found */
      return entryp;
    }

    if (l == h)  return NULL;	/* not found */

    if (nativeoff < entryp->start)
      h = index;
    else
      l = index + 1;
  }
}
#endif	/* EXC_BY_SIGNAL */
