/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

#include <stdlib.h>	/* for getenv(),atoi() */
#include <sys/stat.h>	/* for gdbm_open(), stat() */
#include <unistd.h>	/* for stat() */

#include "compiler.h"

#if JDK_VER >= 12
#  include "jit.h"	/* for JITInterface, JITInterface6 */
#endif

#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
#  include <signal.h>
#endif	/* EXC_BY_SIGNAL || GET_SIGCONTEXT */

#ifdef CODE_DB
#  include <fcntl.h>	/* for O_... */
#  include <dlfcn.h>	/* for dl*() */
#endif	/* CODE_DB */

#ifdef METAVM
#  include "jni.h"
#  include "metavm/metavm.h"	/* for macro METAVM_PKG */
#endif


/*
 * Global Variables
 */
#ifndef IGNORE_DISABLE
bool_t compiler_enabled = TRUE;
#endif	/* IGNORE_DISABLE */

#if (defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)) && defined(SEARCH_SIGCONTEXT)
#  ifdef linux
unsigned short reg_gs;
unsigned short reg_fs;
#  endif
unsigned short reg_es;
unsigned short reg_ds;
#  ifdef __FreeBSD__
unsigned short reg_cs;
unsigned short reg_ss;
#  endif
#endif	/* (EXC_BY_SIGNAL || GET_SIGCONTEXT) && SEARCH_SIGCONTEXT */

#ifdef CODE_DB
#  ifdef GDBM
#    include <gdbm.h>
GDBM_FILE db = NULL;
#  else
#    include <ndbm.h>
DBM *db = NULL;
#  endif
int db_page = -1;
#endif	/* CODE_DB */

struct bool_opt_entry { char *name; int num; };
struct bool_opt_entry bool_opt_entry[] = {
  {"quiet", OPT_QUIET},
  {"outcode", OPT_OUTCODE},
  {"codesize", OPT_CODESIZE},
  {"dontcmplvmclz", OPT_DONTCMPLVMCLZ},
  {"igndisable", OPT_IGNDISABLE},
#ifdef CODE_DB
  {"codedb", OPT_CODEDB},
#endif	/* CODE_DB */
  {"cmplatload", OPT_CMPLATLOAD},
  {NULL, -1}
};
int options = 0;
int opt_cmplthreshold = 0;

void *sym_compileAndInvokeMethod, *sym_invokeJITCompiledMethod;
void *sym_invokeJavaMethod, *sym_invokeSynchronizedJavaMethod;
void *sym_invokeAbstractMethod;
void *sym_invokeNativeMethod, *sym_invokeSynchronizedNativeMethod;
void *sym_invokeJNINativeMethod, *sym_invokeJNISynchronizedNativeMethod;
void *sym_invokeLazyNativeMethod;
#ifdef CODE_DB
#  ifdef GDBM
GDBM_FILE (*sym_dbm_open)(char *,int,int,int,void (*)());
void (*sym_dbm_close)(GDBM_FILE);
int (*sym_dbm_store)(GDBM_FILE,datum,datum,int);
datum (*sym_dbm_fetch)(GDBM_FILE,datum);
void (*sym_dbm_sync)(GDBM_FILE);
#  else
DBM *(*sym_dbm_open)(const char *,int,int);
void (*sym_dbm_close)(DBM *);
int (*sym_dbm_store)(DBM *,datum,datum,int);
datum (*sym_dbm_fetch)(DBM *,datum);
#  endif
#endif	/* CODE_DB */

#ifdef EXC_BY_SIGNAL
static struct sigaction sigActForHandler;
#endif


/*
 * Local Functions
 */
static void initializeClass(ClassClass *);
#ifdef __FreeBSD__
static bool_t signalHandler(int sig, int code, struct sigcontext *uc);
#else
static bool_t signalHandler(int sig, void *info, void *uc);
#endif
#define fillFPUEnv(FPUENV)	asm("fstenv %0" : "=m" (FPUENV))
static void showFPUEnv(FILE *, char *indent, char fpuenv[28]);
static void freeClass(ClassClass *);
static unsigned char *compiledCodePC(JavaFrame *, struct methodblock *);
static bool_t pcInCompiledCode(unsigned char *pc, struct methodblock *);
static JavaFrame *framePrev(JavaFrame *frame, JavaFrame *frame_buf);
static void compilerEnable();
static void compilerDisable();
static bool_t compileClass(ClassClass *);
static bool_t compileClasses(Hjava_lang_String *);
static HObject *compilerCommand(HObject *);

#define INITIALIZE_METHOD(MB) \
  if (OPT_SETQ(OPT_CMPLCLINIT) || strcmp((MB)->fb.name, "<clinit>")) {\
    (MB)->invoker = sym_compileAndInvokeMethod;\
    (MB)->CompiledCode = NULL;\
  }


/*
 * Initialization of the compiler.
 * This method is called by JVM.
 */
void java_lang_Compiler_start(
#if JDK_VER < 12
  void ** vector
#else
  JITInterface *jitinterface
#endif
) {
  static int already_called = 0;

  ExecEnv *ee = EE();
  MONITOR_T key;
  int version;
  int compilerVersion;

#ifdef COMPILE_DEBUG
  printf("java_lang_Compiler_start() called.\n");
  fflush(stdout);
#endif

  key = obj_monitor((JHandle *)classJavaLangClass);
  monitorEnter2(ee, key);

  if (already_called) {
    monitorExit2(ee, key);
    return;
  }
  already_called = 1;
  monitorExit2(ee, key);


  /* version check */

#if JDK_VER < 12
  version = *(int *)vector[0];
#else
  version = *(jitinterface->JavaVersion);
#endif
  compilerVersion = version >> 24;

  if (compilerVersion != COMPILER_VERSION) {
    printf("warning: version num. of compiler interface is not %d: %d\n",
	COMPILER_VERSION, compilerVersion);
  }

#ifdef COMPILE_DEBUG
  {
    int javaVersion, javaMinorVersion;
    javaVersion = (version >> 16) & 0xff;
    javaMinorVersion = version & 0xffff;

    printf("Version num. of class file format: %d.%d\n",
	javaVersion, javaMinorVersion);
  }
#endif	/* COMPILE_DEBUG */


  /* get options */
  {
    char *opts = getenv("JAVA_COMPILER_OPT");
    char *opt;

    if (opts) {
#ifdef RUNTIME_DEBUG
      printf("  JAVA_COMPILER_OPT: %s\n", opts);
      fflush(stdout);
#endif

      opt = strtok(opts, ", ");
      while (opt) {
	int i = 0;
	char *opt_name;
	while (opt_name = bool_opt_entry[i].name, opt_name) {
	  if (!strcmp(opt, opt_name)) {
	    OPT_SET(bool_opt_entry[i].num);
	    if (!OPT_SETQ(OPT_QUIET))  printf(" option: %s\n", opt_name);
	  }
	  i++;
	}
	if (!strncmp(opt, "cmplthreshold=", 14)) {
	  opt_cmplthreshold = atoi(opt+14);
	  if (!OPT_SETQ(OPT_QUIET))
	    printf(" option: cmplthreshold = %d\n", opt_cmplthreshold);
	}
	opt = strtok(NULL, ", ");
      }
    }
  }


  /* show credit */
  if (!OPT_SETQ(OPT_QUIET)) {
    fprintf(stderr, CREDIT);
    fflush(stderr);
  }


  /* initialization */

  /* sysAddDLSegment() some libraries to get addresses of functions */
  /* __divdi3, __moddi3, printf, fflush in libc */
  /* fmod in libm */
  {
    char *libs_orig = getenv("JAVA_COMPILER_LIBS");
    if (!libs_orig)  libs_orig = LIBS_ADDDLSEG;	/* defined in compiler.h */

    if (libs_orig) {
      char *libs;
      char sep, nextsep, *elem, *next;
#define NEXT {\
  char *commap, *barp;\
  elem = next;  sep = nextsep;\
  commap = strchr(elem, ',');  barp = strchr(elem, '|');\
  if (commap)\
    if (barp)\
      if (barp < commap) { next = barp;  nextsep = '|'; }\
      else { next = commap;  nextsep = ','; }\
    else { next = commap;  nextsep = ','; }\
  else\
    if (barp) { next = barp;  nextsep = '|'; }\
    else  { next = strchr(elem, '\0');  nextsep = '\0'; }\
  *next = '\0';  next++;\
}

#define SKIP {\
  char *commap;\
  if (nextsep == ',')  NEXT\
  else {\
    commap = strchr(next, ',');\
    if (commap) {\
      next = commap + 1;  nextsep = ',';\
      NEXT\
    }\
    else  sep = '\0';\
  }\
}

      libs = strdup(libs_orig);

      next = libs;  nextsep = '|';	/* initialize */
      NEXT
      do {
	int loaded = 0;
	do {
	  struct stat buf;
	  if (!stat(elem, &buf)) {
	    if (JVM_LoadLibrary(elem)) {
	      loaded = 1;
	      break;
	    }
	  }
#ifdef COMPILE_DEBUG
	  else {
	    printf("can't stat() file: %s\n", elem);  fflush(stdout);
	  }
#endif
	  NEXT
	} while (sep == '|');

	if (!loaded) {
	  printf("FATAL: JIT compiler can't find required libraries: %s\n",
			libs_orig);
	  JVM_Exit(1);
	}

	SKIP
      } while (sep != '\0');

      sysFree(libs);
    }	/* if (libs_orig) */
  }


#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
  {
    char fpuenv[28];
    fillFPUEnv(fpuenv);
    showFPUEnv(stderr, "  ", fpuenv);
  }
#endif


#if (defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT))&& defined(SEARCH_SIGCONTEXT)
#ifdef linux
  asm("movw %%gs,%0" : "=m" (reg_gs));
  asm("movw %%fs,%0" : "=m" (reg_fs));
#endif
  asm("movw %%es,%0" : "=m" (reg_es));
  asm("movw %%ds,%0" : "=m" (reg_ds));
#ifdef __FreeBSD__
  asm("movw %%cs,%0" : "=m" (reg_cs));
  asm("movw %%ss,%0" : "=m" (reg_ss));
#endif

#ifdef RUNTIME_DEBUG
#ifdef linux
  printf("segment registers (gs,fs,es,ds): %04x, %04x, %04x, %04x\n",
	reg_gs, reg_fs, reg_es, reg_ds);
#endif	/* linux */
#ifdef __FreeBSD__
  printf("segment registers (es,ds,cs,ss): %04x, %04x, %04x, %04x\n",
	reg_es, reg_ds, reg_cs, reg_ss);
#endif	/* __FreeBSD__ */
  fflush(stdout);
#endif
#endif	/* (EXC_BY_SIGNAL || GET_SIGCONTEXT) && SEARCH_SIGCONTEXT */

#ifdef EXC_BY_SIGNAL
  /* prepare struct sigaction for signalHandler() */
#ifdef SA_SIGINFO
  sigActForHandler.sa_handler = NULL;
  sigActForHandler.sa_sigaction =
	(void (*)(int,siginfo_t *,void *))signalHandler;
#else
  sigActForHandler.sa_handler = (void (*)(int))signalHandler;
#endif
  sigemptyset(&sigActForHandler.sa_mask);
  sigaddset(&sigActForHandler.sa_mask, SIGPIPE);
  sigActForHandler.sa_flags = 0;
#ifdef SA_SIGINFO
  sigActForHandler.sa_flags |= SA_SIGINFO;
#endif
#ifdef SA_RESTART
  sigActForHandler.sa_flags |= SA_RESTART;
#endif
#endif


#ifdef RESOLVE_SYMBOL_ON_CODE
  /* resolve function symbols in pre-assembled code (assembledCode()) */
  initFunctionSymbols();
#endif


  /* resolve some symbols */
#define RESOLVE_A_INVOKER(NAME)	\
	sym_##NAME = (void *)symbolInSystemClassLoader(#NAME)

  RESOLVE_A_INVOKER(compileAndInvokeMethod);
  RESOLVE_A_INVOKER(invokeJITCompiledMethod);
  RESOLVE_A_INVOKER(invokeJavaMethod);
  RESOLVE_A_INVOKER(invokeSynchronizedJavaMethod);
  RESOLVE_A_INVOKER(invokeAbstractMethod);
  RESOLVE_A_INVOKER(invokeNativeMethod);
  RESOLVE_A_INVOKER(invokeSynchronizedNativeMethod);
  RESOLVE_A_INVOKER(invokeJNINativeMethod);
  RESOLVE_A_INVOKER(invokeJNISynchronizedNativeMethod);
  RESOLVE_A_INVOKER(invokeLazyNativeMethod);

#ifdef COMPILE_DEBUG
#  define SHOW_A_INVOKER(NAME) \
  printf("  " #NAME ":\t0x%08x\n", (int)sym_##NAME)

  printf("symbols:\n");
  SHOW_A_INVOKER(compileAndInvokeMethod);
  SHOW_A_INVOKER(invokeJITCompiledMethod);
  SHOW_A_INVOKER(invokeJavaMethod);
  SHOW_A_INVOKER(invokeSynchronizedJavaMethod);
  SHOW_A_INVOKER(invokeAbstractMethod);
  SHOW_A_INVOKER(invokeNativeMethod);
  SHOW_A_INVOKER(invokeSynchronizedNativeMethod);
  SHOW_A_INVOKER(invokeJNINativeMethod);
  SHOW_A_INVOKER(invokeJNISynchronizedNativeMethod);
  SHOW_A_INVOKER(invokeLazyNativeMethod);
#endif


  /* initialize the classes which is already loaded */
  if (!OPT_SETQ(OPT_DONTCMPLVMCLZ)) {
    ClassClass **clazzptr;
    int i;

#if JDK_VER < 12
    BINCLASS_LOCK();
#else
    BINCLASS_LOCK(sysThreadSelf());
#endif
#ifdef COMPILE_DEBUG
    printf("%d classes is already loaded.\n", nbinclasses);
    fflush(stdout);
#endif
    for (i = nbinclasses, clazzptr = binclasses; --i >= 0; clazzptr++) {
#ifdef COMPILE_DEBUG
      printf("classes already in VM: %s\n", cbName(*clazzptr));
      fflush(stdout);
#endif
      initializeClass(*clazzptr);
    }
#if JDK_VER < 12
    BINCLASS_UNLOCK();
#else
    BINCLASS_UNLOCK(sysThreadSelf());
#endif
  }


  /* prohibit generating lossy opcodes */
  UseLosslessQuickOpcodes = TRUE;


  /* set up link vector */
#if JDK_VER < 12
  *(char **)vector[1] = (char *)initializeClass;
  *(char **)vector[2] = (char *)sym_invokeJITCompiledMethod;
  *(char **)vector[3] = (char *)signalHandler;
  *(char **)vector[4] = (char *)freeClass;
  *(char **)vector[5] = (char *)compileClass;
  *(char **)vector[6] = (char *)compileClasses;
#ifndef IGNORE_DISABLE
  if (!OPT_SETQ(OPT_IGNDISABLE)) {
    *(char **)vector[7] = (char *)compilerEnable;
    *(char **)vector[8] = (char *)compilerDisable;
  }
#endif	/* IGNORE_DISABLE */
  *(char **)vector[10] = (char *)pcInCompiledCode;
  *(char **)vector[11] = (char *)compiledCodePC;
  *(char **)vector[70] = (char *)framePrev;
#ifdef COMPILE_DEBUG
  {
    int i;
    for (i = 1; i <= 8; i++)
      printf("*vector[%d]: 0x%08x\n", i, (int)*(char **)vector[i]);
    printf("*vector[10]: 0x%08x\n", i, (int)*(char **)vector[10]);
    printf("*vector[11]: 0x%08x\n", i, (int)*(char **)vector[11]);
    printf("*vector[70]: 0x%08x\n", i, (int)*(char **)vector[70]);
    fflush(stdout);
  }
#endif	/* COMPILE_DEBUG */
#else	/* JDK_VER */
  {
    JITInterface6 *ji6 = (JITInterface6 *)jitinterface;

    *ji6->p_InitializeForCompiler = initializeClass;
    *ji6->p_invokeCompiledMethod = sym_invokeJITCompiledMethod;
    *ji6->p_CompiledCodeSignalHandler = signalHandler;
    *ji6->p_CompilerFreeClass = freeClass;

    *ji6->p_CompilerCompileClass = compileClass;
    *ji6->p_CompilerCompileClasses = compileClasses;
#ifndef IGNORE_DISABLE
    if (!OPT_SETQ(OPT_IGNDISABLE)) {
      *ji6->p_CompilerEnable = compilerEnable;
      *ji6->p_CompilerDisable = compilerDisable;
    }
#endif	/* IGNORE_DISABLE */

    *ji6->p_PCinCompiledCode = pcInCompiledCode;
    *ji6->p_CompiledCodePC = compiledCodePC;
    *ji6->p_CompiledFramePrev = framePrev;
  }
#endif	/* JDK_VER */


#ifdef CODE_DB
  /* open DB and page file */
  if (OPT_SETQ(OPT_CODEDB)) {
    void *dl_dbm;
    if (!(dl_dbm = dlopen(LIBDBM, RTLD_LAZY))) {
      fputs(dlerror(), stderr);  fputc('\n', stderr);
      fprintf(stderr, "failed to open " LIBDBM ".\n");
      goto codedb_init_fail;
    }
#ifdef GDBM
    sym_dbm_open = (GDBM_FILE (*)(char *,int,int,int,void (*)()))
	dlsym(dl_dbm, "gdbm_open");
    sym_dbm_close = (void (*)(GDBM_FILE))dlsym(dl_dbm, "gdbm_close");
    sym_dbm_store = (int (*)(GDBM_FILE,datum,datum,int))
	dlsym(dl_dbm, "gdbm_store");
    sym_dbm_fetch = (datum (*)(GDBM_FILE,datum))dlsym(dl_dbm, "gdbm_fetch");
    sym_dbm_sync = (void (*)(GDBM_FILE))dlsym(dl_dbm, "gdbm_sync");
#else
    sym_dbm_open = (DBM *(*)(const char *,int,int))dlsym(dl_dbm, "dbm_open");
    sym_dbm_close = (void (*)(DBM *))dlsym(dl_dbm, "dbm_close");
    sym_dbm_store = (int (*)(DBM *,datum,datum,int))dlsym(dl_dbm, "dbm_store");
    sym_dbm_fetch = (datum (*)(DBM *,datum))dlsym(dl_dbm, "dbm_fetch");
#endif
    if (!(((int32_t)sym_dbm_open) && ((int32_t)sym_dbm_close) &&
	  ((int32_t)sym_dbm_store) && ((int32_t)sym_dbm_fetch))
#ifdef GDBM
	  && ((int32_t)sym_dbm_sync)
#endif
	) {
      fprintf(stderr, "cannot get symbols to handle DBM.\n");
      goto codedb_init_fail;
    }

    if ((db_page = open(CODEDB_PAGE,
		O_RDWR|O_CREAT, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH)) < 0) {
      perror("open");  goto codedb_init_fail;
    }

#ifdef GDBM
    if (!(db = sym_dbm_open(CODEDB_DB, 512,
	GDBM_WRCREAT, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH, NULL))) {
      perror("gdbm_open");  goto codedb_init_fail;
    }
#else
    if (!(db = sym_dbm_open(CODEDB_DB, O_RDWR|O_CREAT, 0))) {
      perror("dbm_open");  goto codedb_init_fail;
    }
#endif

    goto codedb_init_done;
  codedb_init_fail:
    fprintf(stderr, "disable codedb.\n");  OPT_RESET(OPT_CODEDB);
    if (db_page >= 0)  close(db_page);
    JVM_Exit(1);
  codedb_init_done:
  }
#endif	/* CODE_DB */


#ifdef METAVM
  {
    ExecEnv *ee = EE();
    /* force JVM to load Proxy class */
    FindClass(ee, METAVM_PKG "Proxy", TRUE);

    /* set remote flag on */
    REMOTE_FLAG_ON(ee);
  }
#endif


#ifdef COMPILE_DEBUG
  printf("java_lang_Compiler_start() done.\n");
  fflush(stdout);
#endif
}


/*
 * Local Functions
 */

/*
 * Initialize the class at the time one is loaded.
 */
void initializeClass(ClassClass *cb) {
  struct methodblock *mb, *mb_end;
#ifdef COMPILE_DEBUG
  printf("initializeClass(%s) called.\n", cbName(cb));
  fflush(stdout);
#endif

#ifdef METAVM
  /* force a loaded class to implement java.io.Serializable */
  {
    ExecEnv *ee = EE();  JNIEnv *env = EE2JNIEnv(ee);
    static ClassClass *clz_Serializable = NULL;
    if (!clz_Serializable) {
      jclass jclz_Ser = (*env)->FindClass(env, "java/io/Serializable");
      (*env)->NewGlobalRef(env, jclz_Ser);
      clz_Serializable = DeRef(env, jclz_Ser);
    }

    if (!cbIsInterface(cb) && (cb != classJavaLangObject))
      forceToImplement(ee, cb, clz_Serializable);
  }
#endif

  if (OPT_SETQ(OPT_CMPLATLOAD)) {
    compileClass(cb);
    return;
  }

  mb = cbMethods(cb);
  mb_end = mb + cbMethodsCount(cb);
  for (; mb < mb_end; mb++) {
    int access = mb->fb.access;

    if (access & ACC_ABSTRACT)  continue;

    /* initialize mb->CompiledCodeInfo */
    if (!(mb->CompiledCodeInfo))  prepareCompiledCodeInfo(EE(), mb);

    if (access & ACC_NATIVE)  continue;

    INITIALIZE_METHOD(mb);
  }
#ifdef COMPILE_DEBUG
  printf("initializeClass(%s) done.\n", cbName(cb));
  fflush(stdout);
#endif
}


#ifdef __FreeBSD__
static bool_t signalHandler(int sig, int code, struct sigcontext *uc) {
#else
static bool_t signalHandler(int sig, void *info, void *uc) {
#endif
#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
#  ifdef SEARCH_SIGCONTEXT
  static int sc_offset = -1;
  static int sc_search_times = 2;
#if 1
  int32_t *ptr = ((int32_t *)&sig) - 1;
#else
  int32_t *ptr = ((int32_t *)&uc);
#endif
#  endif	/* SEARCH_SIGCONTEXT */
  SIGCONTEXT *sc = NULL;
  struct methodblock *mb;
  CodeInfo *codeinfo;
  uint32_t native_off;
#endif	/* EXC_BY_SIGNAL || GET_SIGCONTEXT */
#ifdef EXC_BY_SIGNAL
  throwentry *tentry;
  uint32_t jump_target = 0;
#endif	/* EXC_BY_SIGNAL */

#if defined(RUNTIME_DEBUG) || defined(COMPILE_DEBUG)
  printf("Signal handler: %d (sys_thread_t: %x)\n", sig, (int)sysThreadSelf());
  showStackFrames(EE());
#endif


#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
  /* get methodblock */
#if 1
  mb = EE()->current_frame->current_method;
	/* EBP is broken on FreeBSD ??? */
#else
  mb = (struct methodblock *) *(((int32_t *)sc->SC_EBP) + 3);
	/* mb is second argument of assembledCode(): 0xc(%ebp) */
#endif
#ifdef RUNTIME_DEBUG
  printf("method(0x%x)", (int)mb);  fflush(stdout);
  fflush(stdout);
  if (!mb)  goto signal_handler_error;
  printf(": %s#%s %s\n",
	cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature);
  fflush(stdout);
#endif
  codeinfo = (CodeInfo *)(mb->CompiledCodeInfo);


#ifndef SEARCH_SIGCONTEXT
  sc = (SIGCONTEXT *)uc;
#else
  /* get signal context */
  if (--sc_search_times < 0) {
    sc = (SIGCONTEXT *)(ptr + sc_offset);
  }
  else {	/* must search */
#define SC_SEARCH_WIDTH	50
    for (sc_offset = 0; sc_offset < SC_SEARCH_WIDTH - 3; sc_offset++) {
#if defined(linux)
      if ( (*((unsigned short *)(ptr)) == reg_gs) &&
	(*((unsigned short *)(ptr + 1)) == reg_fs) &&
	(*((unsigned short *)(ptr + 2)) == reg_es) &&
	(*((unsigned short *)(ptr + 3)) == reg_ds) )
	break;
#elif defined(__FreeBSD__)
      if ( (*((unsigned short *)(ptr)) == reg_es) &&
	(*((unsigned short *)(ptr + 1)) == reg_ds) &&
	(*((unsigned short *)(ptr + 2)) == reg_cs) &&
	(*((unsigned short *)(ptr + 3)) == reg_ss) ) {
	sc_offset -= 7;
	ptr -= 7;
	break;
      }
#else
      if (0)  break;
#endif
      else
	ptr += 1;
    }
#ifdef RUNTIME_DEBUG
    printf("sc_offset: %d\n", sc_offset);
#endif
    if (sc_offset >= (SC_SEARCH_WIDTH - 3)) {
      printf("FATAL: cannot find struct sigcontext on stack.");
      JVM_Exit(1);
    }
    sc = (SIGCONTEXT *)ptr;
  }
#endif	/* SEARCH_SIGCONTEXT*/

#if defined(RUNTIME_DEBUG) || (defined(GET_SIGCONTEXT) && !defined(EXC_BY_SIGNAL))
  printf("sigcontext: 0x%08x\n", (int)sc);  fflush(stdout);
  showSigcontext(sc);
#endif
#endif	/* EXC_BY_SIGNAL */


#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
  native_off =
	((uint32_t)sc->SC_EIP) - ((uint32_t)mb->CompiledCode);
#ifdef RUNTIME_DEBUG
  printf("native off: 0x%x\n", native_off);
  fflush(stdout);
#endif
#ifndef EXC_BY_SIGNAL
  JVM_Exit(1);
#else
  tentry = throwtableGet(codeinfo, native_off);
#ifdef RUNTIME_DEBUG
  printf("throwentry: ");
  fflush(stdout);
  if (tentry)
    printf("start: 0x%x, len: 0x%x, byteoff: 0x%x\n",
		tentry->start, tentry->len, tentry->byteoff);
  else
    printf(" (null)\n");
  fflush(stdout);
#endif	/* RUNTIME_DEBUG */

#define IP_SEARCH_WIDTH	50
  if (!tentry) {	/* search EIP in native code */
    uint32_t *sp = (uint32_t *)sc->SC_ESP;
    uint32_t *bound = sp + IP_SEARCH_WIDTH;

    while (++sp <= bound) {
      native_off = *sp - (uint32_t)mb->CompiledCode;
      if ((native_off >= 0) && (native_off <= codeinfo->code_size)) {
	tentry = throwtableGet(codeinfo, native_off);
	if (tentry) {
#ifdef ARITHEXC_BY_SIGNAL
	  if (sig == SIGFPE) {
	    int opc = tentry->opcode;
	    if ((opc != opc_idiv) && (opc != opc_irem) &&
		(opc != opc_ldiv) && (opc != opc_lrem))
	      continue;
	  }
#endif	/* ARITHEXC_BY_SIGNAL */
	  sc->SC_ESP = (uint32_t)(sp + 1);	/* stack top in the frame */
	  sc->SC_EIP = *sp;			/* %eip on the native code */
	  goto tentry_is_found;
	}
      }
    }
    goto signal_handler_error;
  tentry_is_found:
  }

#ifdef NULLEXC_BY_SIGNAL
  if ((sig == SIGSEGV) || (sig == SIGBUS)) {
	/* NullPointerException occurred */
    SignalError(NULL, JAVAPKG "NullPointerException", 0);
	/* set ee->exception.exc, exception object */
  }
#endif
#if defined(NULLEXC_BY_SIGNAL) && defined(ARITHEXC_BY_SIGNAL)
  else
#endif
#ifdef ARITHEXC_BY_SIGNAL
  if (sig == SIGFPE) {
    if ((tentry->opcode == opc_idiv) || (tentry->opcode == opc_irem)) {
      if (sc->SC_ECX /* divisor */ == 0) {
	/* ArithmeticException occurred */
	SignalError(NULL, JAVAPKG "ArithmeticException", "/ by zero");
      }
      else if ((sc->SC_EAX /* dividend */ == 0x80000000) &&
	       (sc->SC_ECX /* divisor  */ == -1)) {
	/* idiv insn. of x86 causes it: 0x80000000 / -1 */
	sc->SC_EAX = 0x80000000;
	sc->SC_EDX = 0;
	jump_target = sc->SC_EIP + 2;	/* `+ 2' skips idiv `0xf7 0xXX' */
      }
      else {
	float dividend, divisor;
	memcpy(&dividend, &(sc->SC_EAX), 4);
	memcpy(&divisor, &(sc->SC_ECX), 4);
	printf("FATAL: SIGFPE occurred in %s: %e %s %e.\n",
		((tentry->opcode == opc_idiv)?"idiv":"irem"),
		(double)dividend, ((tentry->opcode == opc_idiv)?"/":"%"),
		(double)divisor);
	fflush(stdout);

	goto signal_handler_error;
      }
    }
    else if ((tentry->opcode == opc_ldiv) || (tentry->opcode == opc_lrem)) {
      uint32_t *sp = (uint32_t *)sc->SC_ESP;
      sc->SC_EBP = *(sp + 4);	/* *(sp + 4) is %ebp `back up'ed in ldiv */

      /* dividend: (*(sp + 1) << 32) | *sp
	 divisor : (*(sp + 3) << 32) | *(sp + 2) */

      if (!((*(sp + 2)) | (*(sp + 3)))) {
	/* ArithmeticException occurred */
	SignalError(NULL, JAVAPKG "ArithmeticException", "/ by zero");
      }
#if 0
      /* functions called by ldiv (__divdi3(),__moddi3()) don't cause SIGFPE */
      else if ((*sp == 0) && (*(sp + 1) == 0x80000000) &&
	       ((*(sp +2)  & *(sp + 3)) == 0xffffffff)) {
      }
#endif
    }
    else {
      printf("FATAL: SIGFPE occured in %s.",
	((tentry->opcode == opc_ldiv)?"ldiv":"lrem"));
      fflush(stdout);

      goto signal_handler_error;
    }
  }	/* if (sig == SIGFPE) */
#endif	/* ARITHEXC_BY_SIGNAL */
#if defined(NULLEXC_BY_SIGNAL) || defined(ARITHEXC_BY_SIGNAL)
  else
#endif
  {
    return FALSE;
  }


  /* re-initialize signal handler
     associated with raised signal */
#ifdef __FreeBSD__
  {
    struct sigaction sigAct;
    sigAct = sigActForHandler;
    sigaction(sig, &sigAct, (struct sigaction *)NULL);
  }
#endif	/* __FreeBSD__ */

#ifdef __FreeBSD__
  /* unmask the signal */
  {
    sigset_t sigmask;
    sigemptyset(&sigmask);
    sigaddset(&sigmask, sig);
    sigprocmask(SIG_UNBLOCK, &sigmask, NULL);
  }
#endif	/* __FreeBSD__ */


  /* set (int32_t)bytepcoff (local variable in assembledCode) */
  *(((int32_t *)sc->SC_EBP) - 1) = tentry->byteoff;


  /* JDK 1.2: set program counter to jump to exception handler */
  /* JDK 1.1: jump to exception handler */
#  if JDK_VER >= 12
  {
    uint32_t exc_handler;
    exc_handler = jump_target ? jump_target :
		(uint32_t)(mb->CompiledCode + codeinfo->exc_handler_nativeoff);
    sc->SC_EIP = exc_handler;
  }

  return TRUE;
#  else
  {
    register uint32_t exc_handler asm("edi");
    register SIGCONTEXT *sc_ptr asm("edx");

    exc_handler = jump_target ? jump_target :
		(uint32_t)(mb->CompiledCode + codeinfo->exc_handler_nativeoff);
    sc_ptr = sc;
    asm("movl %0,%%esi" : : "m" (sc_ptr->SC_ESI));
    asm("movl %0,%%ebp" : : "m" (sc_ptr->SC_EBP));
    asm("movl %0,%%esp" : : "m" (sc_ptr->SC_ESP));

    asm("movl %0,%%eax" : : "m" (sc_ptr->SC_EAX));
    asm("movl %0,%%edx" : : "m" (sc_ptr->SC_EDX));

    asm("jmp  *%0" : : "r" (exc_handler));
  }
#  endif	/* JDK_VER */
	/* jump to exception handler */

#endif	/* ! EXC_BY_SIGNAL */
#endif	/* EXC_BY_SIGNAL || GET_SIGCONTEXT */

signal_handler_error:
  fprintf(stderr, "Signal %d\n", sig);
  fflush(stderr);
#if defined(EXC_BY_SIGNAL) || defined(GET_SIGCONTEXT)
  if (sc) {
    struct methodblock *sig_mb = methodByPC((unsigned char *)sc->SC_EIP);
    fprintf(stderr, "  in ");
    if (sig_mb) {
      uint32_t nativepc =
		(uint32_t)sc->SC_EIP - (uint32_t)sig_mb->CompiledCode;
      CodeInfo *info = (CodeInfo *)sig_mb->CompiledCodeInfo;
      fprintf(stderr, "%s#%s %s\n",
		cbName(fieldclass(&sig_mb->fb)),
		sig_mb->fb.name, sig_mb->fb.signature);
      fprintf(stderr, "  native PC: 0x%x(%d)\n", nativepc, nativepc);
      if (info) {
	throwentry *tentry = throwtableGet(info, nativepc);
	if (tentry) {
	  int bytepc = tentry->byteoff;
	  fprintf(stderr, "  byte   PC: 0x%x(%d)\n", bytepc, bytepc);
	}
      }
    }
    {
      char fpuenv[28];
      fillFPUEnv(fpuenv);
      showFPUEnv(stderr, "  ", fpuenv);
    }
    fprintf(stderr, "\n");
    fflush(stderr);

    showSigcontext(sc);

    fprintf(stderr, "\n");
    fflush(stderr);
  }
#endif
  return FALSE;
}


static void showFPUEnv(FILE *fp, char *indent, char fpuenv[28]) {
  if (!indent)  indent = "";
  if (!fp)  fp = stdout;

  fprintf(fp, "%sFPU control word: 0x%04x\n", indent,
		*((int16_t *)(fpuenv + 0)));
  fprintf(fp, "%sFPU status  word: 0x%04x\n", indent,
		*((int16_t *)(fpuenv + 4)));
  fprintf(fp, "%sFPU tag     word: 0x%04x\n", indent,
		*((int16_t *)(fpuenv + 8)));
  fprintf(fp, "%sFPU FIP         : 0x%08x\n", indent,
		*((int32_t *)(fpuenv + 12)));

  fflush(fp);
}


/*
 * Freeing class staffs related to the compiler.
 */
static void freeClass(ClassClass *cb) {
  struct methodblock *mb = cbMethods(cb);
  struct methodblock *mb_end = mb + cbMethodsCount(cb);

#ifdef COMPILE_DEBUG
  printf("freeClass() called.\n");
  fflush(stdout);
#endif

  for (; mb < mb_end; mb++)
    freeMethod(mb);
}


static unsigned char *compiledCodePC(JavaFrame *frame,struct methodblock *mb) {
  return (frame ? frame->lastpc : mb->code);
}


static bool_t pcInCompiledCode(
	unsigned char *pc /* mb->lastpc */, struct methodblock *mb) {
#if 1	/* mb->lastpc is a pc on bytecode */
  unsigned int off = pc - mb->code;
  return (off < mb->code_length);
#else	/* mb->lastpc is a pc on compiled code */
  CodeInfo *info = (CodeInfo *)mb->CompiledCodeInfo;
  if (info) {
    unsigned char *compiled_code = (unsigned char *)mb->CompiledCode;
    unsigned char *limit = compiled_code + info->code_size;
    if (compiled_code != limit) {
      if ((compiled_code <= pc) && (pc < limit))  return TRUE;
    }
  }
  return FALSE;
#endif
}


static JavaFrame *framePrev(JavaFrame *frame, JavaFrame *frame_buf) {
  return frame->prev;
}


#ifndef IGNORE_DISABLE
/*
 * Enable and Disable the compiler.
 */
static void compilerEnable() {
#ifdef RUNTIME_DEBUG
  printf("compilerEnable() called.\n");  fflush(stdout);
#endif
  compiler_enabled = TRUE;
}

/*
 * Enable and Disable the compiler.
 */
static void compilerDisable() {
#ifdef RUNTIME_DEBUG
  printf("compilerDisable() called.\n");  fflush(stdout);
#endif
  compiler_enabled = FALSE;
}
#endif	/* IGNORE_DISABLE */


/*
 * Compile the class. Not only initialization.
 */
static bool_t compileClass(ClassClass *cb) {
  struct methodblock *mb, *mb_end;

#ifdef COMPILE_DEBUG
  printf("\n");
  printf("compileClass(%s) called.\n", cbName(cb));
  fflush(stdout);
#endif

#ifndef IGNORE_DISABLE
  if (!compiler_enabled) {
#ifdef COMPILE_DEBUG
    printf("  compiler has been disabled.\n  return.\n");
    fflush(stdout);
#endif
    return TRUE;
  }
#endif	/* IGNORE_DISABLE */

  mb = cbMethods(cb);
  mb_end = mb + cbMethodsCount(cb);
  for (; mb < mb_end; mb++) {
    int access = mb->fb.access;

    if (access & ACC_ABSTRACT)  continue;

    /* initialize mb->CompiledCodeInfo */
    if (!(mb->CompiledCodeInfo))  prepareCompiledCodeInfo(EE(), mb);

    if (access & ACC_NATIVE)  continue;

    if (!(mb->code)) {
#ifdef COMPILE_DEBUG
      printf("mb->code is NULL: %s#%s %s\n",
		cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature);
      fflush(stdout);
#endif
      INITIALIZE_METHOD(mb);
      continue;
    }

    if (compileMethod(mb))  return FALSE;
  }

#ifdef COMPILE_DEBUG
  printf("compileClass(%s) done.\n", cbName(cb));
  fflush(stdout);
#endif

  return TRUE;
}


/*
 * Compile the classes specified by comma-delimited names.
 */
static bool_t compileClasses(Hjava_lang_String *nameStr) {
  char *names = allocCString(nameStr);
  char *name = names;
  char *p = names;
  bool_t done = FALSE;

#ifdef COMPILE_DEBUG
  printf("compileClasses() called.\n");
  fflush(stdout);
#endif

  if (names) {
    ClassClass *cb;

    do {
      while ((*p != ',') && (*p != '\0'))  p++;
      if (*p == '\0')  done = TRUE;
      else  *p = '\0';

#ifdef COMPILE_DEBUG
      printf("compileClasses(): classname is `%s'\n", name);
      fflush(stdout);
#endif
      cb = FindClass(NULL, name, TRUE);
      if (cb)
	if (!(compileClass(cb)))  goto compileDone;

      p++;
      name = p;
    } while (!done);

  compileDone:
    sysFree(names);

    return TRUE;
  }

  return FALSE;
}


/*
 * Process the command from Java code.
 */
static HObject *compilerCommand(HObject *obj) {
  HObject *ret = NULL;
  
#ifdef COMPILE_DEBUG
  printf("compilerCommand() called.\n");
  fflush(stdout);
#endif

  /* do nothing. */

  return ret;
}
