/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

#include <string.h>	/* for memcpy(), strlen() */
#include <stdlib.h>	/* for malloc() */
#ifdef __FreeBSD__
#  include <sys/types.h>	/* for a type u_long */
#endif
#include <netinet/in.h>	/* for ntohl(), htonl() */

#include "compiler.h"
#include "constants.h"
#include "code.h"	/* for macro STSTA */


#define ALIGNUP32(n)	( ((unsigned int)(n) + 3) & ~((unsigned int)3) )

#define GET_UINT16(p)	((((unsigned char *)(p))[0] << 8) | (p)[1])
#define GET_INT16(p)	((((signed char *)(p))[0] << 8) | (p)[1])
#define GET_INT32(p)\
 ((int)((unsigned char *)(p))[0] << 24| ((int)((unsigned char *)(p))[1] << 16) | ((int)((unsigned char *)(p))[2] << 8) | ((int)((unsigned char *)(p))[3]))


#include <string.h>


/*
 * Local Functions
 */
static int makePCTable(CompilerContext *cc);

static int processAnOpcode(CompilerContext *cc, int opcode,
	int *statep, int32_t *byteoffp);
static int writeCode(CompilerContext *cc);
static void resolveJumpInstructions(CompilerContext *cc);
static void resolveExcRetSw(CompilerContext *cc);

static int resolveDynamicConstants(CompilerContext *cc);
static void makeExcTable(CompilerContext *cc);
static void resolveFunctionSymbols(CompilerContext *cc);


/*
 * Compile a specified method.
 *
 * returns: true if compile failed.
 */
int compileMethod(struct methodblock *mb) {
  CompilerContext *cc;
  CodeInfo *info = (CodeInfo *)mb->CompiledCodeInfo;

#ifdef COMPILE_DEBUG
  printf("\n");
  printf("compileMethod() called.\n  %s#%s %s\n",
	 cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
  {
    int acc = mb->fb.access;
    printf("  access: 0x%x", acc);
    if (acc & ACC_NATIVE)  printf(" native");
/*    if (acc & ACC_MACHINE_COMPILED)  printf(" machine_compiled");*/
    if (acc & ACC_PUBLIC)  printf(" public");
    if (acc & ACC_PRIVATE)  printf(" private");
    if (acc & ACC_PROTECTED)  printf(" protected");
    if (acc & ACC_STATIC)  printf(" static");
    if (acc & ACC_FINAL)  printf(" final");
    if (acc & ACC_SYNCHRONIZED)  printf(" synchronized");
    if (acc & ACC_ABSTRACT)  printf(" abstract");
    printf("\n");
  }
  fflush(stdout);
#endif


  if ((mb->fb.access & (ACC_ABSTRACT | ACC_NATIVE)) ||
	(mb->invoker == sym_invokeJITCompiledMethod)) {
    /* needless to compile */
    return 0;
  }


  cc = getCompilerContext(mb);
#ifdef COMPILE_DEBUG
  showCompilerContext(cc);
#endif

#ifdef CODE_DB
  if (OPT_SETQ(OPT_CODEDB)) {
    if (readCompiledCode(db, db_page, cc))
      goto static_part_done;
  }	/* if (OPT_CODEDB) */
#endif	/* CODE_DB */

  /* compile */
  if (makePCTable(cc))  goto compile_failed;	/* an exception occurred. */
  if (writeCode(cc))  goto compile_failed;	/* an exception occurred. */
  resolveJumpInstructions(cc);
  resolveExcRetSw(cc);

  /* set generated code to methodblock */
  {
    int code_size = cc->bufp - cc->buffer;
#ifdef COMPILE_DEBUG
    printf("compileMethod(): generated code size = 0x%x(%d)\n",
	code_size, code_size);
#endif
    info->code_size = code_size;

    {
      char *code = (void *)sysMalloc(code_size);
      memcpy(code, cc->buffer, code_size);
      mb->CompiledCode = code;
    }
#ifdef COMPILE_DEBUG
    printf("  mb->CompiledCode: 0x%08x\n", (int)mb->CompiledCode);
    fflush(stdout);
#endif
  }

#ifdef CODE_DB
  if (OPT_SETQ(OPT_CODEDB))
    writeCompiledCode(db, db_page, cc);
#endif	/* CODE_DB */

static_part_done:


  resolveDynamicConstants(cc);
  makeExcTable(cc);
  resolveFunctionSymbols(cc);

  mb->invoker =
	(bool_t (*)(JHandle *, struct methodblock *, int, struct execenv *))
		sym_invokeJITCompiledMethod;
			/* invokeJITCompiledMethod() is in invoker.c */
#ifdef CONTROL_COMPILED_FLAG
  mb->fb.access |= ACC_MACHINE_COMPILED;
#endif


  /* write code size */

  if (OPT_SETQ(OPT_CODESIZE)) {
    static FILE *sizefp = NULL;

    if (!sizefp) {
      if (!(sizefp = fopen(CODESIZE_FNAME, "w"))) {
	perror("fopen");
	OPT_RESET(OPT_CODESIZE);
	goto codesize_open_failed;
      }

      fprintf(sizefp, "# num_of_byte_code_inst"
		" size_of_byte_code"
		" size_of_native_code"
		" class_name#method_name signature\n");
    }

    fprintf(sizefp, "%d", cc->pctablelen - 3);
	/* # of byte code inst */
	/* 3 are method{head,tail}, exc. handler. */
    fprintf(sizefp, "\t%lu", mb->code_length);
	/* size of byte code of the method */
    fprintf(sizefp, "\t%d", info->code_size);
	/* size of generated native code */
    fprintf(sizefp, "\t%s#%s %s",
		cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
	/* class name, method name, signature */
    fprintf(sizefp, "\n");
  }
codesize_open_failed:


  releaseCompilerContext(cc);


  /* write generated code to the file code_<classname><methodname>.c */

  if (OPT_SETQ(OPT_OUTCODE)) {
    FILE *codefp;
    char funcname[256];
    char fname[256];
    unsigned char *p;
    int i;

    snprintf(funcname, 256, "%s_%s%s",
	cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
    for (p = funcname; *p != '\0'; p++) {
      switch (*p) {
      case '(':
      case ')':
	*p = '_';  break;
      case '/':
      case ';':  case '<':  case '>':
	*p = '_';  break;
      case '[':  *p = '_';  break;
      }
    }

    snprintf(fname, 256, "code-%s.S", funcname);
    if (!(codefp = fopen(fname, "w"))) {
      perror("fopen");  goto code_open_failed;
    }

    fprintf(codefp, ".globl %s\n", funcname);
#ifdef linux
    fprintf(codefp, ".align 16,0x90\n");
#endif
    fprintf(codefp, "%s:", funcname);
    fprintf(codefp, ".byte ");
    p = (unsigned char *)mb->CompiledCode;
    fprintf(codefp, "%d", p[0]);
    for (i = 1; i < info->code_size; i++)  fprintf(codefp, ",%d", p[i]);
    fprintf(codefp, "\n");

    fclose(codefp);
  }
code_open_failed:


#ifdef COMPILE_DEBUG
  printf("compileMethod() done.\n  %s#%s %s\n",
	cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
  printf("  CompiledCode: 0x%08x\n", (int)mb->CompiledCode);
  fflush(stdout);
#endif

  return 0;


compile_failed:
#ifdef COMPILE_DEBUG
  printf("compileMethod() failed.\n  %s#%s %s\n",
	cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
  fflush(stdout);
#endif

#ifdef CONTROL_COMPILED_FLAG
  mb->fb.access &= ~ACC_MACHINE_COMPILED;
#endif
  /* mb->invoker = access2invoker(mb->fb.access); */	/* needless */

  releaseCompilerContext(cc);

  return 1;
}


/*
 * Freeing method related stuffs.
 */
void freeMethod(struct methodblock *mb) {
  void *info, *code;
#ifdef COMPILE_DEBUG
  printf("freeMethod(): ");
  if (mb)
    printf("%s#%s %s\n",
	cbName(fieldclass(&mb->fb)), mb->fb.name, mb->fb.signature);
  else
    printf("(null)\n");
  fflush(stdout);
#endif

  freeCompiledCodeInfo(mb->CompiledCodeInfo);
  mb->CompiledCodeInfo = NULL;

  code = mb->CompiledCode;
  if (code)  sysFree(code);
  mb->CompiledCode = NULL;
}


/*
 * 1st pass of compilation.
 * generate native code from byte code.
 *
 * returns: true if an exception occurred.
 */
static int makePCTable(CompilerContext *cc) {
  struct methodblock *mb = cc->mb;
  unsigned char *methodcode = mb->code;
  int endoff = mb->code_length;
  int32_t dummy_byteoff;
  int32_t byteoff = 0;

  CodeInfo *info = (CodeInfo *)(mb->CompiledCodeInfo);
#if 0
  int32_t invocation_count = info->invocation_count;
#endif
  int remaked = 0;	/* makePCTable() is retried once at least */

  int opcode;
  int state = 0;

makepctable_start:
  dummy_byteoff = -1;
  if (processAnOpcode(cc, opc_methodhead, &state, &dummy_byteoff))
    return 1;

  while (byteoff < endoff) {
    opcode = *(methodcode + byteoff);

    /* decompose an opcode to some micro opcodes */
    switch (opcode) {
    case opc_iaload:  case opc_laload:  case opc_faload:  case opc_daload:
    case opc_aaload:  case opc_baload:  case opc_caload:  case opc_saload:
      if (processAnOpcode(cc, opc_fill_cache, &state, &byteoff))
	return 1;
      if (processAnOpcode(cc, opc_array_check, &state, &byteoff))
	return 1;
      if (processAnOpcode(cc, opcode, &state, &byteoff))
	return 1;
      break;
    case opc_lstore:  case opc_dstore:
      if (processAnOpcode(cc, opc_fill_cache, &state, &byteoff))
	return 1;
      if (processAnOpcode(cc, opcode, &state, &byteoff))
	return 1;
      break;
    case opc_iastore:  case opc_fastore:
    case opc_aastore:  case opc_bastore:  case opc_castore:  case opc_sastore:
      if (processAnOpcode(cc, opc_iastore1, &state, &byteoff))
	return 1;
      if (processAnOpcode(cc, opc_fill_cache, &state, &byteoff))
	return 1;
      if (processAnOpcode(cc, opc_array_check, &state, &byteoff))
	return 1;
      if (processAnOpcode(cc, opcode, &state, &byteoff))
	return 1;
      break;
    case opc_lastore:  case opc_dastore:
      if (processAnOpcode(cc, opc_fill_cache, &state, &byteoff))
	return 1;
      if (processAnOpcode(cc, opcode, &state, &byteoff))
	return 1;
      break;
    default:
      if (processAnOpcode(cc, opcode, &state, &byteoff))
	return 1;
      break;
    }

#if 0	/* This condition will never happen */
    if (!remaked && (info->invocation_count > invocation_count)) {
	/* The method being compiled is called. */
	/* It's possible that bytecode of the method is changed ! */
#ifdef COMPILE_DEBUG
      printf("restart makePCTable()\n"
	"  because the method is called during being compiled: %s#%s %s\n",
	cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature);
      fflush(stdout);
#endif

      byteoff = 0;
      pctableClear(cc);

      remaked = 1;
      goto makepctable_start;
    }
#endif
  }

  dummy_byteoff = ~(1 << 31);
  /* exception handler */
  if (processAnOpcode(cc, opc_exception_handler, &state, &dummy_byteoff))
    return 1;

  /* method tail */
  if (processAnOpcode(cc, opc_methodtail, &state, &dummy_byteoff))
    return 1;

  return 0;
}


/*
 * returns: true if an exception occurred.
 */
static int processAnOpcode(CompilerContext *cc, int opcode,
	int *statep, int32_t *byteoffp) {
  int state = *statep;

  struct methodblock *mb = cc->mb;
  cp_item_type *constant_pool = cbConstantPool(fieldclass(&mb->fb));
  unsigned char *type_table =
	constant_pool[CONSTANT_POOL_TYPE_TABLE_INDEX].type;

  unsigned char *bytepc = mb->code + *byteoffp;
  int code_opcode;
  CodeTable *codep;

#ifdef COMPILE_DEBUG
  printf("procAnOpc(): %s(0x%02x,%d) st:%d\n",
	((opcode > opc_nonnull_quick) ? "(null)" : opcode_symbol[opcode]),
	opcode, opcode, state);
  printf("  b off: 0x%x(%d)\n", *byteoffp, *byteoffp);
  fflush(stdout);
#endif

  if (!mb->code) {
#ifdef COMPILE_DEBUG
    printf("  mb->code is NULL. cannot compile.\n");  fflush(stdout);
#endif
    return 1;	/* failure */
  }

  /* find the opcode for native code */
  switch (opcode) {
  case opc_aconst_null:
  case opc_fconst_0:
    code_opcode = opc_iconst_0;  break;
  case opc_dconst_0:
    code_opcode = opc_lconst_0;  break;
  case opc_sipush:
  case opc_ldc:
  case opc_ldc_w:
  case opc_ldc_quick:
  case opc_ldc_w_quick:
    code_opcode = opc_bipush;  break;
  case opc_ldc2_w_quick:
    code_opcode = opc_ldc2_w;  break;

  case opc_fload:
  case opc_aload:
    code_opcode = opc_iload;  break;
  case opc_dload:
    code_opcode = opc_lload;  break;

  case opc_fload_0:
  case opc_aload_0:
    code_opcode = opc_iload_0;  break;
  case opc_fload_1:
  case opc_aload_1:
    code_opcode = opc_iload_1;  break;
  case opc_fload_2:
  case opc_aload_2:
    code_opcode = opc_iload_2;  break;
  case opc_fload_3:
  case opc_aload_3:
    code_opcode = opc_iload_3;  break;
  case opc_dload_0:
    code_opcode = opc_lload_0;  break;
  case opc_dload_1:
    code_opcode = opc_lload_1;  break;
  case opc_dload_2:
    code_opcode = opc_lload_2;  break;
  case opc_dload_3:
    code_opcode = opc_lload_3;  break;

  case opc_faload:
  case opc_aaload:
    code_opcode = opc_iaload;  break;
  case opc_daload:
    code_opcode = opc_laload;  break;

  case opc_fstore:
  case opc_astore:
    code_opcode = opc_istore;  break;
  case opc_dstore:
    code_opcode = opc_lstore;  break;

  case opc_fstore_0:
  case opc_astore_0:
    code_opcode = opc_istore_0;  break;
  case opc_fstore_1:
  case opc_astore_1:
    code_opcode = opc_istore_1;  break;
  case opc_fstore_2:
  case opc_astore_2:
    code_opcode = opc_istore_2;  break;
  case opc_fstore_3:
  case opc_astore_3:
    code_opcode = opc_istore_3;  break;
  case opc_dstore_0:
    code_opcode = opc_lstore_0;  break;
  case opc_dstore_1:
    code_opcode = opc_lstore_1;  break;
  case opc_dstore_2:
    code_opcode = opc_lstore_2;  break;
  case opc_dstore_3:
    code_opcode = opc_lstore_3;  break;

  case opc_fastore:
    code_opcode = opc_iastore;  break;
  case opc_dastore:
    code_opcode = opc_lastore;  break;

  case opc_if_acmpeq:
    code_opcode = opc_if_icmpeq;  break;
  case opc_if_acmpne:
    code_opcode = opc_if_icmpne;  break;

  case opc_freturn:
  case opc_areturn:
    code_opcode = opc_ireturn;  break;
  case opc_dreturn:
    code_opcode = opc_lreturn;  break;

  case opc_ifnull:
    code_opcode = opc_ifeq;  break;
  case opc_ifnonnull:
    code_opcode = opc_ifne;  break;

  case opc_goto_w:
    code_opcode = opc_goto;  break;
  case opc_jsr_w:
    code_opcode = opc_jsr;  break;

#define ADJUST_GETMEMBER(vop) \
  case opc_##vop##:\
    {\
      unsigned index = GET_UINT16(bytepc + 1);\
      struct fieldblock *fb;\
      char *sig;\
      \
      if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {\
	if (!ResolveClassConstantFromClass2(\
		fieldclass(&mb->fb), index,\
		cc->ee, 1 << CONSTANT_Fieldref, FALSE))\
	  return 1;\
      }\
      fb = constant_pool[index].fb;\
      sig = fieldsig(fb);\
      code_opcode =\
	((sig[0] == SIGNATURE_LONG) || (sig[0] == SIGNATURE_DOUBLE)) ?\
		opc_##vop##2 : opc_##vop;\
    }\
    break

  ADJUST_GETMEMBER(getstatic);

  ADJUST_GETMEMBER(putstatic);

  case opc_getfield_quick_w:
  ADJUST_GETMEMBER(getfield);

  case opc_putfield_quick_w:
  ADJUST_GETMEMBER(putfield);

  case opc_getfield_quick:
    code_opcode = opc_getfield;  break;
  case opc_putfield_quick:
    code_opcode = opc_putfield;  break;
  case opc_getfield2_quick:
    code_opcode = opc_getfield2;  break;
  case opc_putfield2_quick:
    code_opcode = opc_putfield2;  break;
  case opc_getstatic_quick:
    code_opcode = opc_getstatic;  break;
  case opc_putstatic_quick:
    code_opcode = opc_putstatic;  break;
  case opc_getstatic2_quick:
    code_opcode = opc_getstatic2;  break;
  case opc_putstatic2_quick:
    code_opcode = opc_putstatic2;  break;

  case opc_invokevirtual_quick_w:
    code_opcode = opc_invokevirtual;  break;
  case opc_invokenonvirtual_quick:
  case opc_invokesuper_quick:
    code_opcode = opc_invokespecial;  break;
  case opc_invokestatic_quick:
    code_opcode = opc_invokestatic;  break;
  case opc_invokeinterface_quick:
    code_opcode = opc_invokeinterface;  break;

  case opc_anewarray_quick:
    code_opcode = opc_anewarray;  break;
  case opc_multianewarray_quick:
    code_opcode = opc_multianewarray;  break;
  case opc_checkcast_quick:
    code_opcode = opc_checkcast;  break;
  case opc_instanceof_quick:
    code_opcode = opc_instanceof;  break;

  case opc_wide:
    code_opcode = *(bytepc + 1);  break;

  default:
    code_opcode = opcode;  break;
  }	/* switch (opcode) */

  /* adjust to invokespecial if invokevirtual private method */
  if (opcode == opc_invokevirtual) {
    unsigned index = GET_UINT16(bytepc + 1);
    struct methodblock *method;
    if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {
      int res_result;
#if COMPILE_DEBUG
      printf("  resolving constant_pool[%d]...\n", index);
      fflush(stdout);
#endif
      res_result = ResolveClassConstantFromClass2(
		fieldclass(&mb->fb), index,
		cc->ee, 1 << CONSTANT_Methodref, FALSE);
#if COMPILE_DEBUG
      printf("  resolving constant_pool[%d] done.\n", index);
      if (exceptionOccurred(cc->ee)) {
	JHandle *exc = cc->ee->exception.exc;
	printf("exception occurred: %s\n",
		cbName(exc->methods->classdescriptor));
	showExcStackTrace(exc);
      }
      fflush(stdout);
#endif
      if (!res_result)  return 1;
    }
    method = constant_pool[index].mb;

    if (method->fb.access & ACC_PRIVATE) {
#ifdef COMPILE_DEBUG
      printf("  adjust invokevirtual private method to invokespecial.\n");
#endif
      code_opcode = opc_invokespecial;
    }
  }

#if 1
  /* translate invokevirtual Object#<init> to invokeignored_quick */
  if ((code_opcode == opc_invokespecial) ||
	(code_opcode == opc_invokestatic)) {
    int32_t index;
    struct methodblock *method;

    index = GET_UINT16(bytepc + 1);
    if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {
      int res_result;
#if COMPILE_DEBUG
      printf("  resolving constant_pool[%d]...\n", index);
      fflush(stdout);
#endif
      res_result = ResolveClassConstantFromClass2(fieldclass(&mb->fb),
		index, cc->ee, CONSTANT_Methodref, FALSE);
#if COMPILE_DEBUG
      printf("  resolving constant_pool[%d] done.\n", index);
      if (exceptionOccurred(cc->ee)) {
	JHandle *exc = cc->ee->exception.exc;
	printf("exception occurred: %s\n",
		cbName(exc->methods->classdescriptor));
	showExcStackTrace(exc);
      }
      fflush(stdout);
#endif
      if (!res_result)  return 1;
    }
    method = constant_pool[index].mb;

    if (!(method->fb.access & (ACC_NATIVE | ACC_ABSTRACT)) &&
	(method->code[0] == opc_return)) {
      int toCheck = ((method->fb.access & ACC_STATIC) == 0);
#ifdef NO_CHECK
      bytepc[0] = code_opcode = opc_invokeignored_nocheck;
#else
      if (toCheck)
	bytepc[0] = code_opcode = opc_invokeignored_quick;
      else
	bytepc[0] = code_opcode = opc_invokeignored_nocheck;
#endif	/* NO_CHECK */

      bytepc[1] = method->args_size;
      bytepc[2] = (unsigned char)toCheck;
	/* indicates whether to be checked or not */
    }
  }
#endif	/* #if 0 or 1 */

#ifdef COMPILE_DEBUG
  printf("  adj opc 0x%02x(%d) to 0x%02x(%d)\n",
		opcode, opcode, code_opcode, code_opcode);
  fflush(stdout);
#endif


  /* register opcode, bytepc, state to the table */
  pctableAdd(cc, code_opcode, state, *byteoffp, -1);
  codep = &code_table[code_opcode][state];
#ifdef COMPILE_DEBUG
  printf("  next st:%d\n", codep->after_state);  fflush(stdout);
#endif


  /*
   * update bytecode PC and state
   */

  /* update bytecode PC */
  if (opcode <= opc_nonnull_quick) {
    switch (opcode) {
    case opc_tableswitch:
      {
	int32_t *argp = (int32_t *)ALIGNUP32(bytepc + 1);
	int32_t l = (int32_t)ntohl((unsigned long)argp[1]);
	int32_t h = (int32_t)ntohl((unsigned long)argp[2]);

	argp += 3 + (h - l + 1);
	*byteoffp += ((unsigned char *)argp) - bytepc;
      }
      break;
    case opc_lookupswitch:
      {
	int32_t *argp = (int32_t *)ALIGNUP32(bytepc + 1);
	int32_t npairs = (int32_t)ntohl((unsigned long)argp[1]);

	argp += 2 + (npairs * 2);
	*byteoffp += ((unsigned char *)argp) - bytepc;
      }
      break;
    case opc_wide:
      if (*(bytepc + 1) == opc_iinc)
	*byteoffp += 6;
      else
	*byteoffp += 4;
      break;
    default:
      *byteoffp += opcode_length[opcode];
      break;
    }	/* switch (opcode) */
  }	/* if (opcode <= opc_nonnull_quick) */


  /* update state */
  state = codep->after_state;
  if (state != STSTA) {
    switch (code_opcode) {
	/* jump instructions */
    case opc_goto:
    case opc_jsr:
    case opc_ret:
    case opc_tableswitch:
    case opc_lookupswitch:
      *statep = STATE_AFTER_JUMP;
      break;
    default:
      *statep = state;
      break;
    }
  }

  return 0;
}


static void makeExcTable(CompilerContext *cc) {
  struct CatchFrame_w_state *cf;
  int i;
#ifdef COMPILE_DEBUG
  printf("makeExcTable() called.\n");
  fflush(stdout);
#endif

  cf = (CatchFrame_w_state *)cc->mb->exception_table;
  for (i = 0; i < cc->mb->exception_table_length; i++) {
    pcentry *table = pctableGet(cc, cf->handler_pc);
    int nativeoff = table->nativeoff;
		/* offset in byte code -> offset in native code */
    cf->compiled_CatchFrame = (void *)nativeoff;
    cf->state = table->state;

    cf++;
  }
#ifdef COMPILE_DEBUG
  printf("makeExcTable() done.\n");
  fflush(stdout);
#endif
}


static int writeCode(CompilerContext *cc) {
  struct methodblock *mb = cc->mb;
  CodeInfo *info = (CodeInfo *)(mb->CompiledCodeInfo);
  cp_item_type *constant_pool = cbConstantPool(fieldclass(&mb->fb));
  unsigned char *type_table =
	constant_pool[CONSTANT_POOL_TYPE_TABLE_INDEX].type;

  pcentry *pctable;
  int opcode, state;
  int32_t nativeoff = 0, nextoff = 0;
  unsigned char *bytepc;
  int i;

  CodeTable *codep;
  unsigned char *bufp;  int insn_head_off;
#ifdef COMPILE_DEBUG
  printf("writeCode called: %s#%s %s.\n",
	cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature);
  printf("  type_table: 0x%x\n", (int)type_table);
  fflush(stdout);
#endif

  for (i = 0; i < cc->pctablelen; i++) {
    pctable = cc->pctable + i;
    opcode = pctable->opcode;
    state = pctable->state;
    bytepc = mb->code + pctable->byteoff;

#ifdef COMPILE_DEBUG
    printf("writeCode(): %s(0x%02x,%d) st:%d\n",
	((opcode > opc_nonnull_quick) ? "(null)" : opcode_symbol[opcode]),
	opcode, opcode, state);
    printf("  off: b 0x%x(%d) n 0x%x(%d)\n",
		pctable->byteoff, pctable->byteoff,
		nativeoff, nativeoff);
    fflush(stdout);
#endif

    codep = &code_table[opcode][state];
#ifdef COMPILE_DEBUG
    printf("  len native: 0x%x(%d)\n", codep->length, codep->length);
    fflush(stdout);
#endif

    if (opcode == opc_exception_handler) {
      info->exc_handler_nativeoff = nativeoff;
    }
    else if (opcode == opc_methodtail) {
      info->finish_return_nativeoff = nativeoff;
    }

    insn_head_off = cc->bufp - cc->buffer;
	/* save to resolve constants */

    /* write native code to buffer */
    {
      unsigned char *nativecode =
		(unsigned char *)assembledCode + codep->offset;
      writeToBuffer(cc, nativecode, codep->length);
      pctable->nativeoff = nativeoff;

#ifdef EXC_BY_SIGNAL
      if (nativecode[-1]) {
	/* these native code may throw
		NullPointerException or ArithmeticException */
	throwtableAdd(info, nativeoff, (uint16_t)codep->length,
	  (unsigned char)pctable->byteoff, (unsigned char)opcode);
#ifdef COMPILE_DEBUG
	printf("throwtableAdd(): opc %x, byte 0x%x, native 0x%x - 0x%x\n",
	  opcode, pctable->byteoff, nativeoff, nativeoff + codep->length);
	fflush(stdout);
#endif
      }
#endif	/* EXC_BY_SIGNAL */
    }

    nextoff = nativeoff + codep->length;


    /* register jump instructions to table
       and update nextoff. */
    if ((opc_ifeq <= opcode) && (opcode <= opc_ret))
	/* || ((opc_ifnull <= opcode) && (opcode <= opc_ifnonnull)) */
	/* || (opcode == opc_goto_w) || (opcode == opc_jsr_w)) */ {
      int tgtoff = -1;
      int tgtstate;
      int laststate;
      CodeTable *transcodep;

      laststate = code_table[opcode][state].after_state;
      if (laststate == STSTA)  laststate = state;

#ifdef COMPILE_DEBUG
      printf("  jump instruction: %s(0x%x,%d)\n",
		opcode_symbol[*bytepc], *bytepc, *bytepc);
      printf("    last state: %d\n", laststate);
      fflush(stdout);
#endif
      if (opcode == opc_ret) {
	tgtstate = STATE_AFTER_JUMP;
      }
      else {
	int jumpoff;
	pcentry *tgttable;

	if ((*bytepc == opc_goto_w) || (*bytepc == opc_jsr_w))
	  jumpoff = GET_INT32(bytepc + 1);
	else
	  jumpoff = GET_INT16(bytepc + 1);
	tgtoff = pctable->byteoff + jumpoff;
	tgttable = pctableGet(cc, tgtoff);
	tgtstate = tgttable->state;
#ifdef COMPILE_DEBUG
	{
	  int tgtop = tgttable->opcode;
	  printf("    target %s(0x%x,%d) offset: 0x%x(%d), state: %d\n",
		((tgtop > opc_nonnull_quick)?"(null)":opcode_symbol[tgtop]),
		tgtop, tgtop, tgtoff, tgtoff, tgtstate);
	  fflush(stdout);
	}
#endif
      }
      transcodep = &code_table[opc_stateto0 + tgtstate][laststate];

      if ((opc_ifeq <= opcode) && (opcode <= opc_if_acmpne))
	/* || ((opc_ifnull <= opcode) && (opcode <= opc_ifnonnull)) */ {
	int rop = 0, rop_rev = 0;

#define JP_CASE(OP, ROP4, ROP_REV1) \
      case opc_if##OP:\
      case opc_if_icmp##OP:\
	rop = ROP4;  rop_rev = ROP_REV1;\
	break
	/* 0x0f, ROP4 takes 4 byte operand, ROP_REV1 takes 1 byte. */

	switch (opcode) {
	  JP_CASE(eq, 0x84/*je*/, 0x75/*jne*/);
	  JP_CASE(ne, 0x85/*jne*/, 0x74/*je*/);
	  JP_CASE(lt, 0x8c/*jl*/, 0x7d/*jge*/);
	  JP_CASE(ge, 0x8d/*jge*/, 0x7c/*jl*/);
	  JP_CASE(gt, 0x8f/*jg*/, 0x7e/*jle*/);
	  JP_CASE(le, 0x8e/*jle*/, 0x7f/*jg*/);
	}

	if (laststate == tgtstate) {
	  unsigned char code_jp[] = { 0x0f, rop, 0, 0, 0, 0 };
	  /* jxx X(4byte) */
	  writeToBuffer(cc, code_jp, 6);
	  nextoff += 6;
	}
	else {
	  /* jXX 5 */
	  {
	    unsigned char code_jp[] = { rop_rev, transcodep->length + 5 };
	    writeToBuffer(cc, code_jp, 2);
	    nextoff += 2;
	  }
	  /* code_statetoX */
	  pctableInsert(cc, i + 1,
		opc_stateto0 + tgtstate, laststate,
		pctable->byteoff, nextoff);
	  i++;
	  writeToBuffer(cc,
		(unsigned char *)assembledCode + transcodep->offset,
		transcodep->length);
	  nextoff += transcodep->length;
	  /* jmp X(4byte) */
	  {
	    unsigned char code_jp[] = { 0xe9, 0, 0, 0, 0 };
	    writeToBuffer(cc, code_jp, 5);
	  }
	  nextoff += 5;
	}

	jptableAdd(cc, tgtoff, nextoff - 4);
      }
      else	/* ((opc_goto <= opcode) || (opcode <= opc_ret)) */
	/* || ((opcode == opc_goto_w) && (opcode == opc_jsr_w)) */ {
	/* code_sattetoX */
	pctableInsert(cc, i + 1,
		opc_stateto0 + tgtstate, laststate,
		pctable->byteoff, nextoff);
	i++;
	writeToBuffer(cc, (unsigned char *)assembledCode + transcodep->offset,
			transcodep->length);
	nextoff += transcodep->length;

	if (opcode != opc_ret) {	/* goto, jsr */
	  /* jmp X(4byte) */
	  unsigned char code_jp[] = { 0xe9, 0, 0, 0, 0 };
	  writeToBuffer(cc, code_jp, 5);
	  nextoff += 5;

	  jptableAdd(cc, tgtoff, nextoff - 4);
	}
	else {	/* ret */
	  /* jmp *%eax */
	  unsigned char code_jp[] = { 0xff, 0xe0 };
	  writeToBuffer(cc, code_jp, 2);
	  nextoff += 2;
	}
      }
    }	/* if jump instruction */


    /* resolve constants */
    bufp = cc->buffer + insn_head_off;

    switch (opcode) {
    case opc_bipush:
      {
	int32_t v;
	switch (*bytepc) {
	case opc_bipush:
	  v = (signed char)bytepc[1];
#ifdef COMPILE_DEBUG
	  printf("  const: %d\n", v);  fflush(stdout);
#endif
	  break;
	case opc_sipush:
	  v = GET_INT16(bytepc + 1);
#ifdef COMPILE_DEBUG
	  printf("  const: %d\n", v);  fflush(stdout);
#endif
	  break;
	case opc_ldc:
	  v = bytepc[1];
#ifdef COMPILE_DEBUG
	  printf("  index: %d\n", v);  fflush(stdout);
#endif
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, v))
	    if (!ResolveClassConstantFromClass2(fieldclass(&mb->fb), v, cc->ee,
			(1 << CONSTANT_Integer) | (1 << CONSTANT_Float) |
			(1 << CONSTANT_String),
			FALSE))
	      return 1;
	  if (CONSTANT_POOL_TYPE_TABLE_GET_TYPE(type_table, v) ==
		CONSTANT_String)
	    goto wc_ldc_break;
	  v = constant_pool[v].i;
	  break;
	case opc_ldc_w:
	  v = GET_UINT16(bytepc + 1);
#ifdef COMPILE_DEBUG
	  printf("  index: %d\n", v);  fflush(stdout);
#endif
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, v))
	    if (!ResolveClassConstantFromClass2(fieldclass(&mb->fb), v, cc->ee,
			(1 << CONSTANT_Integer) | (1 << CONSTANT_Float) |
			(1 << CONSTANT_String),
			FALSE))
	      return 1;
	  if (CONSTANT_POOL_TYPE_TABLE_GET_TYPE(type_table, v) ==
		CONSTANT_String)
	    goto wc_ldc_break;
	  v = constant_pool[v].i;
	  break;
	case opc_ldc_quick:
	  v = bytepc[1];
#ifdef COMPILE_DEBUG
	  printf("  index: %d\n", v);  fflush(stdout);
#endif
	  if (CONSTANT_POOL_TYPE_TABLE_GET_TYPE(type_table, v) ==
		CONSTANT_String)
	    goto wc_ldc_break;
	  v = constant_pool[v].i;
	  break;
	case opc_ldc_w_quick:
	  v = GET_UINT16(bytepc + 1);
#ifdef COMPILE_DEBUG
	  printf("  index: %d\n", v);  fflush(stdout);
#endif
	  if (CONSTANT_POOL_TYPE_TABLE_GET_TYPE(type_table, v) ==
		CONSTANT_String)
	    goto wc_ldc_break;
	  v = constant_pool[v].i;
	  break;
	default:
	  goto wc_ldc_break;
	}

	memcpy(bufp + constant_table[opcode][state][0], &v, 4);
      }
    wc_ldc_break:
      break;

    case opc_ldc2_w:
      {
	int index = GET_UINT16(bytepc + 1);
	int32_t cp_entry1 = constant_pool[index].i;
	int32_t cp_entry2 = constant_pool[index + 1].i;
#ifdef COMPILE_DEBUG
	printf("  index: %d\n", index);  fflush(stdout);
#endif

	if (*bytepc == opc_ldc2_w) {
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {
	    if (!ResolveClassConstantFromClass2(
		fieldclass(&mb->fb), index, cc->ee,
		(1 << CONSTANT_Long) | (1 << CONSTANT_Double),
		FALSE))
	      return 1;
	  }
	  cp_entry1 = constant_pool[index].i;
	  cp_entry2 = constant_pool[index + 1].i;
	}		/* opc_ldc2_w_quick */
	memcpy(bufp + constant_table[opcode][state][0], &cp_entry2, 4);
	memcpy(bufp + constant_table[opcode][state][1], &cp_entry1, 4);
      }
      break;

    case opc_iload:
    case opc_istore:
    case opc_ret:
      {
	int32_t index;
	if (*bytepc == opc_wide)
	  index = GET_UINT16(bytepc + 2);
	else
	  index = bytepc[1];
#ifdef COMPILE_DEBUG
	printf("  index: %d\n", index);  fflush(stdout);
#endif
#ifdef RUNTIME_DEBUG
	memcpy(bufp + constant_table[opcode][state][1], &index, 4);
#endif
	index *= -4;
	memcpy(bufp + constant_table[opcode][state][0], &index, 4);
      }
      break;

    case opc_lload:
    case opc_lstore:
      {
	int32_t index;
	if (*bytepc == opc_wide)
	  index = GET_UINT16(bytepc + 2);
	else
	  index = bytepc[1];
#ifdef COMPILE_DEBUG
	printf("  index: %d\n", index);  fflush(stdout);
#endif
	index = (index + 1) * -4;
	memcpy(bufp + constant_table[opcode][state][0], &index, 4);
#ifdef RUNTIME_DEBUG
	memcpy(bufp + constant_table[opcode][state][2], &index, 4);
#endif
	index += 4;
	memcpy(bufp + constant_table[opcode][state][1], &index, 4);
      }
      break;

#if defined(METAVM) && !defined(METAVM_NO_ARRAY)
    case opc_iaload:	/* includes opc_aaload */
    case opc_baload:
    case opc_caload:
    case opc_saload:
      {
	int32_t *p = (int32_t *)(bufp + constant_table[opcode][state][0]);
	if (*bytepc == opc_aaload) {
	  *p = (int32_t)1;
	}
	else
	  *p = (int32_t)0;
      }
      break;

    case opc_iastore:
    case opc_bastore:
    case opc_castore:
    case opc_sastore:
      {
	int32_t *p = (int32_t *)(bufp + constant_table[opcode][state][0]);
	*p = (int32_t)0;
      }
      break;

    case opc_aastore:
      {
	int32_t *p = (int32_t *)(bufp + constant_table[opcode][state][0]);
	*p = (int32_t)1;
      }
      break;
#endif	/* METAVM_NO_ARRAY */

    case opc_iinc:
      {
	int32_t index, constbyte;
	
	if (*bytepc == opc_wide) {
	  index = GET_UINT16(bytepc + 2);
	  constbyte = GET_INT16(bytepc + 4);
	}
	else {
	  index = bytepc[1];
	  constbyte = ((signed char *)bytepc)[2];
	}
#ifdef COMPILE_DEBUG
	printf("  vars[%d] += %d\n", index, constbyte);  fflush(stdout);
#endif
	index *= -4;
#ifdef RUNTIME_DEBUG
	memcpy(bufp + constant_table[opcode][state][2], &index, 4);
	memcpy(bufp + constant_table[opcode][state][2] + 4, &constbyte, 4);

	memcpy(bufp + constant_table[opcode][state][0], &index, 4);
	memcpy(bufp + constant_table[opcode][state][1], &index, 4);
	memcpy(bufp + constant_table[opcode][state][3], &index, 4);
	memcpy(bufp + constant_table[opcode][state][4], &constbyte, 4);
#else
	memcpy(bufp + constant_table[opcode][state][0], &index, 4);
	memcpy(bufp + constant_table[opcode][state][0] + 4, &constbyte, 4);
#endif
      }
      break;

    case opc_jsr:
#ifdef COMPILE_DEBUG
      printf("  next native offset: 0x%x(%d)\n", nextoff, nextoff);
      fflush(stdout);
#endif
      memcpy(bufp + constant_table[opcode][state][0], &nextoff, 4);
      break;

    case opc_tableswitch:
      {
	int32_t *argp = (int32_t *)ALIGNUP32(bytepc + 1);
	int32_t l = (int32_t)ntohl((unsigned long)argp[1]);
	int32_t h = (int32_t)ntohl((unsigned long)argp[2]);
#ifdef COMPILE_DEBUG
	printf("  low: %d, high: %d\n", l, h);  fflush(stdout);
#endif
	memcpy(bufp + constant_table[opcode][state][0], &l, 4);
	memcpy(bufp + constant_table[opcode][state][1], &h, 4);
      }
      break;

    case opc_lookupswitch:
      {
	int32_t *argp = (int32_t *)ALIGNUP32(bytepc + 1);
	int32_t npairs = (int32_t)ntohl((unsigned long)argp[1]);
#ifdef COMPILE_DEBUG
	printf("  npairs: %d\n", npairs);  fflush(stdout);
#endif
	memcpy(bufp + constant_table[opcode][state][0], &npairs, 4);
      }
      break;

#ifdef COMPILE_DEBUG
#  define CONST_GETMEMBER_DEBUG1 \
	printf("  %s#%s\n", cbName(fb->clazz), fb->name);\
	fflush(stdout)
#  define CONST_GETMEMBER_DEBUG2 printf("  slot: %d\n", slot); fflush(stdout)
#else
#  define CONST_GETMEMBER_DEBUG1
#  define CONST_GETMEMBER_DEBUG2
#endif

#ifdef RUNTIME_DEBUG
#  define CONST_GETMEMBER_DEBUG3\
  memcpy(bufp + constant_table[opcode][state][1], &slot, 4)
#else
#  define CONST_GETMEMBER_DEBUG3
#endif

#ifdef METAVM
	/* supply info. whether if field is object to generated native code */
#  define METAVM_CONST_GETMEMBER(vop) \
	if (opcode == opc_##vop) {\
	  int32_t objp;\
	  switch (*fb->signature) {\
	  case 'L':  case '[':\
	    objp = 1;\
	    break;\
	  default:\
	    objp = 0;\
	    break;\
	  }\
	  memcpy(bufp + constant_table[opcode][state][1], &objp, 4);\
	}
#else
#  define METAVM_CONST_GETMEMBER(vop)
#endif	/* METAVM */

#ifdef RUNTIME_DEBUG
#  define RUNTIME_DEBUGP	1
#else
#  define RUNTIME_DEBUGP	0
#endif
#ifdef METAVM
#  define METAVMP	1
#else
#  define METAVMP	0
#endif	/* METAVM */

#define CONST_GETMEMBER(vop) \
    case opc_##vop##:\
    case opc_##vop##2:\
      {\
	struct fieldblock *fb = NULL;\
	int32_t slot;\
	if ((*bytepc == opc_##vop##_quick) || (*bytepc == opc_##vop##2_quick))\
		/* after lossy translation (which is suppressed) */\
	  slot = bytepc[1];\
	else {	/* ..field, ..field_quick_w */\
	  int index = GET_UINT16(bytepc + 1);\
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {\
	    if (!ResolveClassConstantFromClass2(\
		fieldclass(&mb->fb), index, cc->ee,\
		1 << CONSTANT_Fieldref,\
		FALSE))\
	      return 1;\
	  }\
	  fb = constant_pool[index].fb;\
	  CONST_GETMEMBER_DEBUG1;\
	  slot = fb->u.offset / sizeof(OBJECT);\
	}\
	CONST_GETMEMBER_DEBUG2;\
	memcpy(bufp + constant_table[opcode][state][0], &slot, 4);\
	if (RUNTIME_DEBUGP)\
	  if (METAVMP && (opcode == opc_##vop))\
	    memcpy(bufp + constant_table[opcode][state][2], &slot, 4);\
	  else\
	    memcpy(bufp + constant_table[opcode][state][1], &slot, 4);\
        CONST_GETMEMBER_DEBUG3;\
	\
	METAVM_CONST_GETMEMBER(vop)\
      }\
      break

    CONST_GETMEMBER(getfield);
    CONST_GETMEMBER(putfield);

#ifdef COMPILE_DEBUG
#  define WC_CONST_INVOKE_DEBUG1() \
	printf("  not resolved yet.\n");  fflush(stdout)
#  define WC_CONST_INVOKE_DEBUG2() \
	method = constant_pool[index].mb;\
	printf("  clz initialized: %s\n",\
		(CB_INITIALIZED(method->fb.clazz) ? "true" : "false"));\
	if (!(method->CompiledCodeInfo)) {\
	  printf("  WARNING: method->CompiledCodeInfo is NULL. (writeCode())\n");\
	  fflush(stdout);\
	}
#  define WC_CONST_INVOKE_DEBUG3(VIRTUALP) \
	printf("  %s#%s %s\n", cbName(method->fb.clazz), method->fb.name,\
		method->fb.signature);\
	fflush(stdout);\
	printf("  args_size, ret_size, localvar_space: %d, %d, %d\n",\
		args_size, ret_size, localvar_space);\
	fflush(stdout);\
	if (VIRTUALP)\
	  printf("  offset in table: %d\n", method->fb.u.offset);\
	fflush(stdout)
#else
#  define WC_CONST_INVOKE_DEBUG1()
#  define WC_CONST_INVOKE_DEBUG2()
#  define WC_CONST_INVOKE_DEBUG3(VIRTUALP)
#endif

#define WC_CONST_INVOKE(vop, VIRTUALP, INTERFACEP) \
      {\
	int32_t index;\
	struct methodblock *method;\
	char *sig;\
	int32_t args_size, localvar_space, ret_size;\
	\
	index = GET_UINT16(bytepc + 1);\
	if (*bytepc == opc_invoke##vop) {\
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {\
	    WC_CONST_INVOKE_DEBUG1();\
	    if (!ResolveClassConstantFromClass2(\
		fieldclass(&mb->fb), index, cc->ee,\
		1 << (INTERFACEP ?\
			CONSTANT_InterfaceMethodref : CONSTANT_Methodref),\
		FALSE))\
	      return 1;\
	  }\
	  WC_CONST_INVOKE_DEBUG2();\
	}\
	method = constant_pool[index].mb;\
	args_size = method->args_size;\
	\
	localvar_space = (method->nlocals - args_size);\
	if (localvar_space > 0)  localvar_space *= 4;\
	else  localvar_space = 0;\
		/* nlocals is 0 in the case of native method */\
	\
	sig = method->fb.signature;\
	while (*(sig++) != ')');\
	if (*sig == 'V')  ret_size = 0;\
	else if ((*sig == 'D') || (*sig == 'J'))  ret_size = 2;\
	else  ret_size = 1;\
	\
	WC_CONST_INVOKE_DEBUG3(VIRTUALP);\
	\
	memcpy(bufp + constant_table[opcode][state][0], &args_size, 4);\
	if (VIRTUALP) {		/* invokevirtual */\
	  uint32_t slot = method->fb.u.offset;\
	  memcpy(bufp + constant_table[opcode][state][1], &slot, 4);\
	  memcpy(bufp + constant_table[opcode][state][2], &ret_size, 4);\
	}\
	else if (INTERFACEP) {	/* invokeinterface */\
	  memcpy(bufp + constant_table[opcode][state][3], &ret_size, 4);\
	}\
	else {			/* invokespecial, invokestatic */\
	  memcpy(bufp + constant_table[opcode][state][2], &localvar_space, 4);\
	  memcpy(bufp + constant_table[opcode][state][3], &ret_size, 4);\
	}\
      }\
      break

    case opc_invokevirtual:
      WC_CONST_INVOKE(virtual, 1, 0);

    case opc_invokespecial:
      WC_CONST_INVOKE(special, 0, 0);

    case opc_invokestatic:
      WC_CONST_INVOKE(static, 0, 0);

    case opc_invokeinterface:
      WC_CONST_INVOKE(interface, 0, 1);

    case opc_invokevirtualobject_quick:
    case opc_invokevirtual_quick:
      {
	uint32_t slot = bytepc[1];
	int32_t localvar_space = 0;	/* can't get this information !!! */
	int32_t args_size = bytepc[2];
	int32_t ret_size = 0;		/* can't get this information !!! */

	memcpy(bufp + constant_table[opcode][state][0], &args_size, 4);
	memcpy(bufp + constant_table[opcode][state][1], &localvar_space, 4);
	memcpy(bufp + constant_table[opcode][state][2], &slot, 4);
	memcpy(bufp + constant_table[opcode][state][3], &ret_size, 4);
      }
      break;

    case opc_newarray:
      {
	int32_t type = bytepc[1];
#ifdef COMPILE_DEBUG
	printf("  type: %d\n", type);  fflush(stdout);
#endif
	memcpy(bufp + constant_table[opcode][state][0], &type, 4);
      }
      break;

    case opc_multianewarray:
      {
	int32_t dimensions = bytepc[3];
#ifdef COMPILE_DEBUG
	printf("  index: %d\n  dimensions: %d\n", index, dimensions);
	fflush(stdout);
#endif
	memcpy(bufp + constant_table[opcode][state][0], &dimensions, 4);
      }
      break;

    case opc_invokeignored_quick:
    case opc_invokeignored_nocheck:
      {
	int32_t args_size = bytepc[1];
#ifdef COMPILE_DEBUG
	printf("  args_size: %d\n", args_size);  fflush(stdout);
#endif
	memcpy(bufp + constant_table[opcode][state][0], &args_size, 4);
      }
      break;
    }	/* resolve constants: switch (opcode) */

    /* update native offset */
    nativeoff = nextoff;
  }	/* for (i = 0; i < cc->pctablelen; i++) */

#ifdef COMPILE_DEBUG
  printf("writeCode done: %s#%s %s.\n",
	cbName(mb->fb.clazz), mb->fb.name, mb->fb.signature);
  fflush(stdout);
#endif
  return 0;
}


static int resolveDynamicConstants(CompilerContext *cc) {
  struct methodblock *mb = cc->mb;
  CodeInfo *info = (CodeInfo *)(mb->CompiledCodeInfo);
  cp_item_type *constant_pool = cbConstantPool(fieldclass(&mb->fb));
  unsigned char *type_table =
	constant_pool[CONSTANT_POOL_TYPE_TABLE_INDEX].type;

  pcentry *pctable;
  int opcode, state;
  int32_t nativeoff = 0;
  unsigned char *bytepc;
  int i;

  CodeTable *codep;
  unsigned char *bufp;

  for (i = 0; i < cc->pctablelen; i++) {
    pctable = cc->pctable + i;
    opcode = pctable->opcode;
    state = pctable->state;
    bytepc = mb->code + pctable->byteoff;
    nativeoff = pctable->nativeoff;

#ifdef COMPILE_DEBUG
    printf("resolveDynConst(): %s(0x%02x,%d) st:%d\n",
	((opcode > opc_nonnull_quick) ? "(null)" : opcode_symbol[opcode]),
	opcode, opcode, state);
    printf("  off: b 0x%x(%d) n 0x%x(%d)\n",
		pctable->byteoff, pctable->byteoff,
		nativeoff, nativeoff);
    fflush(stdout);
#endif

    codep = &code_table[opcode][state];
#ifdef COMPILE_DEBUG
    printf("  len native: 0x%x(%d)\n", codep->length, codep->length);
    fflush(stdout);
#endif

    /* resolve constants */
	/* attention: var. nativeoff is already updated. */
#if 1
    bufp = mb->CompiledCode + nativeoff;
#else
    bufp = cc->buffer + nativeoff;
#endif

    switch (opcode) {
    case opc_bipush:
      {
	int32_t v;
	switch (*bytepc) {
	case opc_ldc:
	  v = bytepc[1];
#ifdef COMPILE_DEBUG
	  printf("  index: %d\n", v);  fflush(stdout);
#endif
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, v))
	    if (!ResolveClassConstantFromClass2(fieldclass(&mb->fb), v, cc->ee,
			(1 << CONSTANT_Integer) | (1 << CONSTANT_Float) |
			(1 << CONSTANT_String),
			FALSE))
	      return 1;
	  if (CONSTANT_POOL_TYPE_TABLE_GET_TYPE(type_table, v) !=
		CONSTANT_String)
	    goto rc_ldc_break;
	  v = constant_pool[v].i;
	  break;
	case opc_ldc_w:
	  v = GET_UINT16(bytepc + 1);
#ifdef COMPILE_DEBUG
	  printf("  index: %d\n", v);  fflush(stdout);
#endif
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, v))
	    if (!ResolveClassConstantFromClass2(fieldclass(&mb->fb), v, cc->ee,
			(1 << CONSTANT_Integer) | (1 << CONSTANT_Float) |
			(1 << CONSTANT_String),
			FALSE))
	      return 1;
	  if (CONSTANT_POOL_TYPE_TABLE_GET_TYPE(type_table, v) !=
		CONSTANT_String)
	    goto rc_ldc_break;
	  v = constant_pool[v].i;
	  break;
	case opc_ldc_quick:
	  v = bytepc[1];
#ifdef COMPILE_DEBUG
	  printf("  index: %d\n", v);  fflush(stdout);
#endif
	  if (CONSTANT_POOL_TYPE_TABLE_GET_TYPE(type_table, v) !=
		CONSTANT_String)
	    goto rc_ldc_break;
	  v = constant_pool[v].i;
	  break;
	case opc_ldc_w_quick:
	  v = GET_UINT16(bytepc + 1);
#ifdef COMPILE_DEBUG
	  printf("  index: %d\n", v);  fflush(stdout);
#endif
	  if (CONSTANT_POOL_TYPE_TABLE_GET_TYPE(type_table, v) !=
		CONSTANT_String)
	    goto rc_ldc_break;
	  v = constant_pool[v].i;
	  break;
	default:
	  goto rc_ldc_break;
	}

	memcpy(bufp + constant_table[opcode][state][0], &v, 4);
      }
    rc_ldc_break:
      break;

    case opc_getstatic:
    case opc_putstatic:
      {
	unsigned index = GET_UINT16(bytepc + 1);
	struct fieldblock *fb;
	OBJECT *addr;
#ifdef COMPILE_DEBUG
	printf("  index: %d\n", index);  fflush(stdout);
#endif
	if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {
	  if (!ResolveClassConstantFromClass2(
		fieldclass(&mb->fb), index, cc->ee,
		1 << CONSTANT_Fieldref,
		FALSE))
	    return 1;
	}
	fb = constant_pool[index].fb;
#ifdef COMPILE_DEBUG
	printf("  %s#%s\n", cbName(fb->clazz), fb->name);  fflush(stdout);
#endif
	{
	  ClassClass *cb = fb->clazz;
#if defined(INITCLASS_IN_COMPILING) || defined(NO_REWRITE)
#  define RC_STATIC_OFFSET 0
	  if (!CB_INITIALIZED(cb))  InitClass(cb);
#else
#  define RC_STATIC_OFFSET 1
	  memcpy(bufp + constant_table[opcode][state][0], &cb, 4);
#endif
	}
	addr = (OBJECT *)normal_static_address(fb);
	memcpy(bufp + constant_table[opcode][state][RC_STATIC_OFFSET + 0],
		&addr, 4);
      }
      break;

    case opc_getstatic2:
    case opc_putstatic2:
      {
	int32_t index = GET_UINT16(bytepc + 1);
	struct fieldblock *fb;
	stack_item *addr;
#ifdef COMPILE_DEBUG
	printf("  index: %d\n", index);  fflush(stdout);
#endif
	if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {
	  if (!ResolveClassConstantFromClass2(
		fieldclass(&mb->fb), index, cc->ee,
		1 << CONSTANT_Fieldref,
		FALSE))
	    return 1;
	}
	fb = constant_pool[index].fb;
#ifdef COMPILE_DEBUG
	printf("  %s#%s\n", cbName(fb->clazz), fb->name);  fflush(stdout);
#endif
	{
	  ClassClass *cb = fb->clazz;
#if defined(INITCLASS_IN_COMPILING) || defined(NO_REWRITE)
#  define RC_STATIC2_OFFSET 0
	  if (!CB_INITIALIZED(cb))  InitClass(cb);
#else
#  define RC_STATIC2_OFFSET 1
	  memcpy(bufp + constant_table[opcode][state][0], &cb, 4);
#endif
	}
	addr = (stack_item *)twoword_static_address(fb);
	memcpy(bufp + constant_table[opcode][state][RC_STATIC2_OFFSET + 0],
		&addr, 4);
	addr++;
	memcpy(bufp + constant_table[opcode][state][RC_STATIC2_OFFSET + 1],
		&addr, 4);
      }
      break;

#if defined(INITCLASS_IN_COMPILING) || defined(NO_REWRITE)
#  define CALL_INITCLASS_INVOKE \
	{\
	  ClassClass *cb = method->fb.clazz;\
	  if (!CB_INITIALIZED(cb))  InitClass(cb);\
	}
#else
#  define CALL_INITCLASS_INVOKE
#endif

#ifdef COMPILE_DEBUG
#  define RC_CONST_INVOKE_DEBUG1 \
	printf("  %s#%s %s\n", cbName(method->fb.clazz), method->fb.name,\
		method->fb.signature);\
	fflush(stdout)
#else
#  define RC_CONST_INVOKE_DEBUG1
#endif

#define RC_CONST_INVOKE(vop, VIRTUALP, INTERFACEP) \
      {\
	unsigned index;\
	struct methodblock *method;\
	\
	index = GET_UINT16(bytepc + 1);\
	if (*bytepc == opc_invoke##vop) {\
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {\
	    if (!ResolveClassConstantFromClass2(\
		fieldclass(&mb->fb), index, cc->ee,\
		1 << (INTERFACEP ?\
			CONSTANT_InterfaceMethodref : CONSTANT_Methodref),\
		FALSE))\
	      return 1;\
	  }\
	}\
	method = constant_pool[index].mb;\
	\
	RC_CONST_INVOKE_DEBUG1;\
	\
	CALL_INITCLASS_INVOKE;\
	\
	if (VIRTUALP) {		/* invokevirtual */\
	}\
	else if (INTERFACEP) {	/* invokeinterface */\
	  unsigned char *guessptr = bytepc + 4;\
	  memcpy(bufp + constant_table[opcode][state][1], &guessptr, 4);\
	  memcpy(bufp + constant_table[opcode][state][2], &method, 4);\
	}\
	else {			/* invokespecial, invokestatic */\
	  memcpy(bufp + constant_table[opcode][state][1], &method, 4);\
	}\
      }\
      break

    case opc_invokespecial:
      RC_CONST_INVOKE(special, 0, 0);

    case opc_invokestatic:
      RC_CONST_INVOKE(static, 0, 0);

    case opc_invokeinterface:
      RC_CONST_INVOKE(interface, 0, 1);

#ifdef COMPILE_DEBUG
#  define CONST_NEW_DEBUG1 printf("  index: %d\n", index); fflush(stdout);
#  define CONST_NEW_DEBUG2 printf("  name: %s\n", cbName(cb)); fflush(stdout);
#else
#  define CONST_NEW_DEBUG1
#  define CONST_NEW_DEBUG2
#endif

#if defined(INITCLASS_IN_COMPILING) || defined(NO_REWRITE)
#  define CALL_INITCLASS_NEW \
	if (!CB_INITIALIZED(cb))  InitClass(cb);
#else
#  define CALL_INITCLASS_NEW
#endif

#define _CONST_NEW(vop, RESOLVE) \
	int32_t index = GET_UINT16(bytepc + 1);\
	ClassClass *cb;\
	CONST_NEW_DEBUG1;\
	\
	if (RESOLVE) {\
	  if (*bytepc == opc_##vop) {\
	    if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {\
	      if (!ResolveClassConstantFromClass2(fieldclass(&mb->fb),\
			index, cc->ee, 1 << CONSTANT_Class, FALSE))\
		return 1;\
	    }\
	  }\
	}\
	cb = constant_pool[index].clazz;\
	CALL_INITCLASS_NEW;\
	CONST_NEW_DEBUG2;
#define CONST_NEW1(vop, RESOLVE) \
      {\
	_CONST_NEW(vop, RESOLVE);\
	memcpy(bufp + constant_table[opcode][state][0], &cb, 4);\
      }
#define CONST_NEW2(vop, RESOLVE) \
      {\
	_CONST_NEW(vop, RESOLVE);\
	memcpy(bufp + constant_table[opcode][state][0], &cb, 4);\
	memcpy(bufp + constant_table[opcode][state][1], &cb, 4);\
      }

    case opc_new:
      CONST_NEW1(new, 1);
      break;

    case opc_new_quick:
      CONST_NEW1(new, 0);
      break;

    case opc_anewarray:
#if defined(METAVM) && !defined(METAVM_NO_ARRAY)
      CONST_NEW2(anewarray, 1);
#else
      CONST_NEW1(anewarray, 1);
#endif	/* METAVM_NO_ARRAY */
      break;

    case opc_checkcast:
      CONST_NEW1(checkcast, 1);
      break;

    case opc_instanceof:
      CONST_NEW1(instanceof, 1);
      break;

    case opc_multianewarray:
      {
	int32_t index = GET_UINT16(bytepc + 1);
	ClassClass *cb;
	if (*bytepc == opc_multianewarray) {
	  if (!CONSTANT_POOL_TYPE_TABLE_IS_RESOLVED(type_table, index)) {
	    int res_result;
#if COMPILE_DEBUG
	    printf("  resolving constant_pool[%d]...\n", index);
	    fflush(stdout);
#endif
	    res_result = ResolveClassConstantFromClass2(
			fieldclass(&mb->fb), index,
			cc->ee, 1 << CONSTANT_Class,
			FALSE);
#if COMPILE_DEBUG
	    printf("  resolving constant_pool[%d] done.\n", index);
	    if (exceptionOccurred(cc->ee)) {
	      JHandle *exc = cc->ee->exception.exc;
	      printf("exception occurred: %s\n",
			cbName(exc->methods->classdescriptor));
	      showExcStackTrace(exc);
	    }
	    fflush(stdout);
#endif
	    if (!res_result)  return 1;
	  }
	}
	cb = constant_pool[index].clazz;
#ifdef COMPILE_DEBUG
	printf("  clazz: %s\n", cbName(cb));  fflush(stdout);
#endif
	memcpy(bufp + constant_table[opcode][state][1], &cb, 4);
#if defined(METAVM) && !defined(METAVM_NO_ARRAY)
	memcpy(bufp + constant_table[opcode][state][2], &cb, 4);
#endif	/* METAVM_NO_ARRAY */
      }
      break;

    }	/* resolve constants: switch (opcode) */
  }	/* for (i = 0; i < cc->pctablelen; i++) */

#ifdef COMPILE_DEBUG
  printf("resolveDynConst() done.\n");
  fflush(stdout);
#endif
  return 0;
}


static void resolveJumpInstructions(CompilerContext *cc) {
  unsigned char *nativeCode = cc->buffer;

  jpentry *jptable = cc->jptable;
  jpentry *jptable_end = jptable + cc->jptablelen;

  while (jptable < jptable_end) {
    unsigned int argoff = jptable->argoff;
    pcentry *tgttable = pctableGet(cc, jptable->tgtoff);
    unsigned int tgtoff = tgttable->nativeoff;
    int32_t arg = tgtoff - (argoff + 4);
#ifdef COMPILE_DEBUG
    printf("resolveJpInst(): arg. offset: 0x%x(%d)\n", argoff, argoff);
    printf("  target offset: byte 0x%x(%d), native 0x%x(%d)\n",
		jptable->tgtoff, jptable->tgtoff, tgtoff, tgtoff);
    fflush(stdout);
#endif
    memcpy(nativeCode + argoff, &arg, 4);

    jptable++;
  }
}


/*
 * 2nd pass of compilation.
 * set absolute/relative addresses into compiled code.
 */
static void resolveExcRetSw(CompilerContext *cc) {
  CodeInfo *info = (CodeInfo *)(cc->mb->CompiledCodeInfo);
  unsigned char *nativeCode = cc->buffer;

  pcentry *pctable = cc->pctable;
  pcentry *pctable_end = pctable + cc->pctablelen;

  int i;

  while (pctable < pctable_end) {
    int opcode = pctable->opcode;
    int state = pctable->state;
#ifdef COMPILE_DEBUG
    {
      int byteoff = pctable->byteoff;
      int nativeoff = pctable->nativeoff;

      printf("resERS(): %s(0x%02x,%d) st:%d\n",
	((opcode > opc_nonnull_quick)?"(null)":opcode_symbol[opcode]),
	opcode, opcode, state);
      printf("  off: b 0x%x(%d) n 0x%x(%d)\n",
	byteoff, byteoff, nativeoff, nativeoff);
      fflush(stdout);
    }
#endif

    /* resolve PC in bytecode */
    i = 0;
    if (bytepc_table[opcode][state][i] > 0) {
      uint32_t byteoff = pctable->byteoff;
      while (bytepc_table[opcode][state][i] > 0) {
#ifdef COMPILE_DEBUG
	printf("  res. bytecode PC at 0x%x(%d) + %d: 0x%x(%d)\n",
		pctable->nativeoff, pctable->nativeoff,
		bytepc_table[opcode][state][i], byteoff, byteoff);
	fflush(stdout);
#endif
	memcpy(nativeCode + pctable->nativeoff +
		bytepc_table[opcode][state][i], &byteoff, 4);
	i++;
      }
    }

    /* resolve jump to exception handler */
    i = 0;
    if (jumpexc_table[opcode][state][i] > 0) {
      unsigned int exc_off = info->exc_handler_nativeoff;
      unsigned int arg_off;
      int32_t relative_off;
      while (jumpexc_table[opcode][state][i] > 0) {
	arg_off = pctable->nativeoff + jumpexc_table[opcode][state][i];
	relative_off = exc_off - (arg_off + 4);
#ifdef COMPILE_DEBUG
	printf("  res. jump to exc.handler at 0x%x(%d) + %d\n",
		pctable->nativeoff, pctable->nativeoff,
		jumpexc_table[opcode][state][i]);
	fflush(stdout);
#endif
	memcpy(nativeCode + arg_off, &relative_off, 4);
	i++;
      }
    }

    /* resolve jump to finishing of method */
    i = 0;
    if (jumpret_table[opcode][state][i] > 0) {
      unsigned int fin_off = info->finish_return_nativeoff;
      unsigned int arg_off;
      int32_t relative_off;
      while (jumpret_table[opcode][state][i] > 0) {
	arg_off = pctable->nativeoff + jumpret_table[opcode][state][i];
	relative_off = fin_off - (arg_off + 4);
#ifdef COMPILE_DEBUG
	printf("  res. jump to fin.of method at 0x%x(%d) + %d\n",
		pctable->nativeoff, pctable->nativeoff,
		jumpret_table[opcode][state][i]);
	fflush(stdout);
#endif
	memcpy(nativeCode + arg_off, &relative_off, 4);
	i++;
      }
    }


    /* make a table of native offset */
    switch (opcode) {
    case opc_tableswitch:
      {
	unsigned int byteoff = pctable->byteoff;
	pcentry *tgttable;
	CodeTable *codep;

	int32_t *argp = (int32_t *)ALIGNUP32(cc->mb->code + byteoff + 1);
	int32_t defoff = (int32_t)ntohl((unsigned long)argp[0]);
	int32_t l = (int32_t)ntohl((unsigned long)argp[1]);
	int32_t h = (int32_t)ntohl((unsigned long)argp[2]);
	int32_t *tblp;
	int after_state = code_table[opcode][state].after_state;
	int i;
#ifdef COMPILE_DEBUG
	printf("  low: %d, high: %d\n", l, h);  fflush(stdout);
#endif

	cc->bufp = (unsigned char *)ALIGNUP32(cc->bufp);

	ensureBufferSize(cc, (h - l + 2) * 8);
	nativeCode = cc->buffer;

	tblp = (int32_t *)(cc->bufp);

	/* set table offset into compiled code */
	{
	  int32_t tbloff = ((unsigned char *)tblp) - nativeCode;
	  memcpy(nativeCode + pctable->nativeoff
		+ constant_table[opcode][state][2], &tbloff, 4);
	}

	/* default */
	tgttable = pctableGet(cc, byteoff + defoff);
	tblp[0] = tgttable->nativeoff;
	codep = &code_table[opc_goto_st0 + tgttable->state][after_state];
	tblp[1] = (int32_t)(((unsigned char *)assembledCode) + codep->offset);
#ifdef COMPILE_DEBUG
	{
	  unsigned int boff = byteoff + defoff;
	  printf("  default: byte 0x%x(%d), native 0x%x(%d)\n",
		boff, boff, tblp[0], tblp[0]);
	  fflush(stdout);
	}
#endif
	tblp += 2;
	argp += 3;	/* skit default, low and high */

	for (i = 0; i < (h - l + 1); i++) {
	  int32_t off = (int32_t)ntohl((unsigned long)argp[i]);
	  tgttable = pctableGet(cc, byteoff + off);
	  tblp[0] = tgttable->nativeoff;
	  codep = &code_table[opc_goto_st0 + tgttable->state][after_state];
	  tblp[1] =
		(int32_t)(((unsigned char *)assembledCode) + codep->offset);
#ifdef COMPILE_DEBUG
	  {
	    unsigned int boff = byteoff + off;
	    printf("  %3d: offset: byte 0x%x(%d), native 0x%x(%d)\n",
			l + i, boff, boff, tblp[0], tblp[0]);
	    fflush(stdout);
	  }
#endif
	  tblp += 2;
	}

	cc->bufp = (unsigned char *)tblp;
      }
      break;
    case opc_lookupswitch:
      {
	unsigned int byteoff = pctable->byteoff;
	pcentry *tgttable;
	CodeTable *codep;

	int32_t *argp = (int32_t *)ALIGNUP32(cc->mb->code + byteoff + 1);
	int32_t defoff = (int32_t)ntohl((unsigned long)argp[0]);
	int32_t npairs = (int32_t)ntohl((unsigned long)argp[1]);
	int32_t *tblp;
	int after_state = code_table[opcode][state].after_state;

#ifdef COMPILE_DEBUG
	printf("  npairs: %d\n", npairs);  fflush(stdout);
#endif

	cc->bufp = (unsigned char *)ALIGNUP32(cc->bufp);

	ensureBufferSize(cc, (npairs + 1) * 12);
	nativeCode = cc->buffer;

	tblp = (int32_t *)(cc->bufp);

	/* set table offset int compiled code */
	{
	  int32_t tbloff = ((unsigned char *)tblp) - nativeCode;
	  memcpy(nativeCode + pctable->nativeoff
		+ constant_table[opcode][state][1], &tbloff, 4);
	}

	argp += 2;	/* skip default and npairs */

	/* (int32_t)match and (int32_t)jump offset */
	while (npairs-- > 0) {
	  int32_t match = (int32_t)ntohl((unsigned long)argp[0]);
	  int32_t off = (int32_t)ntohl((unsigned long)argp[1]);

	  tblp[0] = match;
	  tgttable = pctableGet(cc, byteoff + off);
	  tblp[1] = tgttable->nativeoff;
	  codep = &code_table[opc_goto_st0 + tgttable->state][after_state];
	  tblp[2] =
		(int32_t)(((unsigned char *)assembledCode) + codep->offset);
#ifdef COMPILE_DEBUG
	  {
	    unsigned int boff = byteoff + off;
	    printf("  match: %d, offset: byte 0x%x(%d), native 0x%x(%d)\n",
			match, boff, boff, tblp[1], tblp[1]);
	    fflush(stdout);
	  }
#endif
	  argp += 2;
	  tblp += 3;
	}

	/* (int32_t)dummy_match and (int32_t)default */
	tgttable = pctableGet(cc, byteoff + defoff);
	tblp[0] = 0;
	tblp[1] = tgttable->nativeoff;
	codep = &code_table[opc_goto_st0 + tgttable->state][after_state];
	tblp[2] = (int32_t)(((unsigned char *)assembledCode) + codep->offset);
#ifdef COMPILE_DEBUG
	{
	  unsigned int boff = byteoff + defoff;
	  printf("  default offset: byte 0x%x(%d), native 0x%x(%d)\n",
			boff, boff, tblp[1], tblp[1]);
	  fflush(stdout);
	}
#endif
	tblp += 3;

	cc->bufp = (unsigned char *)tblp;
      }
      break;
    }

    pctable++;
  }
}


#ifdef RESOLVE_SYMBOL_ON_CODE
#undef INITFUNC_DEBUG
void initFunctionSymbols() {
  FuncTable *funcp;
  CodeTable *codep;
  int opcode, state;
#ifdef INITFUNC_DEBUG
  printf("initFuncSym() called.\n");  fflush(stdout);
#endif

  for (opcode = 0; opcode < NOPCODES; opcode++) {
    for (state = 0; state < NSTATES; state++) {
      funcp = func_table[opcode][state];
      codep = &code_table[opcode][state];
      while (funcp->offset >= 0) {
	unsigned char *funcptr;
	unsigned char *argptr;
	funcptr = (unsigned char *)symbolInSystemClassLoader(funcp->name);
#ifdef INITFUNC_DEBUG
	printf("  0x%02x(%d),%d: `%s' offset %d\n", opcode, opcode, state,
		funcp->name, funcp->offset);
	printf("    write 0x%08x to .. + 0x%x(%d) + %d\n",
		(int)funcptr, codep->offset, codep->offset, funcp->offset);
	fflush(stdout);
#endif
	argptr =
	  ((unsigned char *)assembledCode) + codep->offset + funcp->offset;
	if (!((opc_goto_st0 <= opcode) && (opcode < opc_goto_st0 + NSTATES))) {
		/* native code for the opcodes is never copied
			and is executed at original place */
	  memcpy(argptr, &funcptr, 4);
	}
	funcp++;
      }
    }	/* for (state) */
  }	/* for (opcode) */
#ifdef INITFUNC_DEBUG
  printf("initFuncSym() done.\n\n");  fflush(stdout);
#endif
}
#endif RESOLVE_SYMBOL_ON_CODE


#undef RES_FUNC_DEBUG
static void resolveFunctionSymbols(CompilerContext *cc) {
  unsigned char *nativeCode = cc->mb->CompiledCode;

  pcentry *pctable = cc->pctable;
  pcentry *pctable_end = pctable + cc->pctablelen;

  FuncTable *funcp;
#ifdef RES_FUNC_DEBUG
  printf("resolveFuncSym() called.\n");  fflush(stdout);
#endif

  while (pctable < pctable_end) {
    int opcode = pctable->opcode;
    int state = pctable->state;

    funcp = func_table[opcode][state];

    while (funcp->offset >= 0) {
      unsigned char *argoff;
      unsigned char *funcptr;
      int32_t arg;

      argoff = nativeCode + pctable->nativeoff + funcp->offset;
#ifdef RESOLVE_SYMBOL_ON_CODE
      memcpy(&funcptr, argoff, 4);
#else
      funcptr = (unsigned char *)symbolInSystemClassLoader(funcp->name);
#ifdef RUNTIME_DEBUG
      {
	ExecEnv *ee = EE();
	if (exceptionOccurred(ee)) {
	  JHandle *exc = ee->exception.exc;
	  printf("exception occurred: %s\n",
		cbName(exc->methods->classdescriptor));
	  showExcStackTrace(exc);
	  fflush(stdout);
	}
      }
#endif
      if (!funcptr) {
	/* not reached. */
	fprintf(stderr, "FATAL: symbol `%s' was not resolved.\n",
			funcp->name);
	JVM_Exit(1);
      }
#endif	/* RESOLVE_SYMBOL_ON_CODE */

#ifdef RES_FUNC_DEBUG
      printf("  symb %3d:%d %s: 0x%08x at 0x%x(%d) + 0x%x\n",
		opcode, state,
		funcp->name, (int)funcptr,
		pctable->nativeoff, pctable->nativeoff, funcp->offset);
      fflush(stdout);
#endif
      arg = funcptr - (argoff + 4);
      memcpy(argoff, &arg, 4);

      funcp++;
    }

    pctable++;
  }
#ifdef RES_FUNC_DEBUG
  printf("resolveFuncSym() done.\n");  fflush(stdout);
#endif
}
