/*
  This file is part of shuJIT,
  Just In Time compiler for Sun Java Virtual Machine.

  Copyright (C) 1998,1999 SHUDO Kazuyuki

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id$
*/

package NET.shudo.metavm;

import java.io.Serializable;
import java.net.InetAddress;
import java.net.Socket;
import java.io.IOException;
import java.io.OutputStream;
import java.io.InputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedInputStream;


/**
 * An instance of this class represents a reference to remote object.
 */
public class Proxy implements ByValue, Cloneable {
  private VMAddress vmaddr;

  private Class clazz;
  private int id;
  private int length;

  transient private Socket sock = null;
  transient private DistObjectOutputStream out;
  transient private DistObjectInputStream in;

  /** debug flag. */
  transient private boolean debug = false;


  static {
    initNative();
  }
  private static native void initNative();


  protected Object clone() throws CloneNotSupportedException {
    return new Proxy(vmaddr, clazz, id, length);
  }


  /**
   * Retuns VM address where this proxy stays on.
   */
  protected VMAddress address() { return vmaddr; }


  private Proxy(VMAddress addr, Class clazz, int id, int length) {
    this.vmaddr = addr;
    this.clazz = clazz;
    this.id = id;
    this.length = length;

    this.debug = MetaVM.debug;
  }
  private Proxy(VMAddress addr, Class clazz) { this(addr, clazz, -1, 0); }
  private Proxy(VMAddress addr) { this(addr, null, -1, 0); }


  /**
   * Make a proxy object correspoinding to a supplied local object.
   */
  public static Proxy get(Object obj) {
    Proxy proxy;
    int length = 0;
    boolean orig = MetaVM.remoteTransparency(false);

    if (obj.getClass().isArray()) {
      length = java.lang.reflect.Array.getLength(obj);
    }

    proxy = new Proxy(VMAddress.localAddress(),
			obj.getClass(), ObjectID.idByObject(obj), length);
    if (MetaVM.debug) {
      System.out.println("  vmaddr: " + proxy.vmaddr);
      System.out.println("  obj: " + obj);
      System.out.println("  id : " + proxy.id);
    }

    ExportTable.register(obj);

    MetaVM.remoteTransparency(orig);

    return proxy;
  }

  /**
   * bytecode insn. `new'.
   */
  public static Proxy get(Class fromClazz, VMAddress addr, Class clazz)
	throws IOException {
		// don't make use of formClazz now.
    Proxy proxy;  int id;
    DistObjectOutputStream out;  DistObjectInputStream in;
    boolean orig = MetaVM.remoteTransparency(false);

    if ((addr == null) || (clazz == null))
      throw new IOException("VM address and class must be specified.");

    proxy = new Proxy(addr, clazz);
    proxy.establishConnection();

    out = proxy.out;  in = proxy.in;

    out.writeByte(Protocol.NEW);

    VMAddress classsrc = null;
    ClassLoader cl = clazz.getClassLoader();
    if ((cl != null) && (cl instanceof RemoteClassLoader))
      classsrc = ((RemoteClassLoader)cl).sourceAddress();
    else
      classsrc = VMAddress.localAddress();

    if (MetaVM.debug) {
      System.out.println("Proxy.get() classsrc is " + classsrc + ".");
      System.out.flush();
    }

    out.writeObject(classsrc);
    out.writeUTF(clazz.getName());
    out.flush();

    id = in.readInt();
    if (id == 0)
      throw new IOException("instantiation failure: " + clazz.getName());
    proxy.id = id;

    MetaVM.remoteTransparency(orig);

    return proxy;
  }

  /**
   * bytecode insn. `newarray'.
   */
  public static Proxy get(Class fromClazz, VMAddress addr, int type, int count)
	throws IOException {
    Proxy proxy;  int id;
    DistObjectOutputStream out;  DistObjectInputStream in;
    boolean orig = MetaVM.remoteTransparency(false);

    proxy = new Proxy(addr);
    proxy.establishConnection();

    out = proxy.out;  in = proxy.in;

    out.writeByte(Protocol.NEWARRAY);

    VMAddress classsrc = null;
    ClassLoader cl = fromClazz.getClassLoader();
    if ((cl != null) && (cl instanceof RemoteClassLoader))
      classsrc = ((RemoteClassLoader)cl).sourceAddress();
    else
      classsrc = VMAddress.localAddress();

    if (MetaVM.debug) {
      System.out.println("Proxy.get() classsrc is " + classsrc + ".");
      System.out.flush();
    }

    out.writeObject(classsrc);
    out.writeByte(type);
    out.writeInt(count);
    out.flush();

    proxy.clazz = TypeUtil.arrayType(TypeUtil.primTypeByCode(type));

    id = in.readInt();
    if (id == 0)
      throw new IOException("instantiation failure: " + proxy.clazz.getName());
    proxy.id = id;
    proxy.length = count;

    MetaVM.remoteTransparency(orig);

    return proxy;
  }

  /**
   * bytecode insn. `anewarray'.
   */
  public static Proxy get(Class fromClazz, VMAddress addr,
				Class clazz, int count) throws IOException {
    Proxy proxy;  int id;
    DistObjectOutputStream out;  DistObjectInputStream in;
    boolean orig = MetaVM.remoteTransparency(false);

    proxy = new Proxy(addr);
    proxy.establishConnection();

    out = proxy.out;  in = proxy.in;

    out.writeByte(Protocol.ANEWARRAY);

    VMAddress classsrc = null;
    ClassLoader cl = fromClazz.getClassLoader();
    if ((cl != null) && (cl instanceof RemoteClassLoader))
      classsrc = ((RemoteClassLoader)cl).sourceAddress();
    else
      classsrc = VMAddress.localAddress();

    if (MetaVM.debug) {
      System.out.println("Proxy.get() classsrc is " + classsrc + ".");
      System.out.flush();
    }

    out.writeObject(classsrc);
    out.writeUTF(clazz.getName());
    out.writeInt(count);
    out.flush();

    proxy.clazz = TypeUtil.arrayType(clazz);

    id = in.readInt();
    if (id == 0)
      throw new IOException("instantiation failure: " + clazz.getName());
    proxy.id = id;
    proxy.length = count;

    MetaVM.remoteTransparency(orig);

    return proxy;
  }

  /**
   * bytecode insn. `multianewarray'.
   */
  public static Proxy get(Class fromClazz, VMAddress addr,
			Class clazz, int[] sizes) throws IOException {
    Proxy proxy;  int id;
    DistObjectOutputStream out;  DistObjectInputStream in;
    boolean orig = MetaVM.remoteTransparency(false);

    proxy = new Proxy(addr);
    proxy.establishConnection();

    out = proxy.out;  in = proxy.in;

    out.writeByte(Protocol.MULTIANEWARRAY);

    VMAddress classsrc = null;
    ClassLoader cl = fromClazz.getClassLoader();
    if ((cl != null) && (cl instanceof RemoteClassLoader))
      classsrc = ((RemoteClassLoader)cl).sourceAddress();
    else
      classsrc = VMAddress.localAddress();

    out.writeObject(classsrc);
    out.writeUTF(clazz.getName());
    out.writeObject(new ArrayOfIntWrapper(sizes));
    out.flush();

    proxy.clazz = TypeUtil.arrayType(clazz);

    id = in.readInt();
    if (id == 0)
      throw new IOException("instantiation failure: " + clazz.getName());
    proxy.id = id;
    proxy.length = sizes[0];

    MetaVM.remoteTransparency(orig);

    return proxy;
  }


  /**
   * Make a connection and get a reference to a remote object
   * correspoinding to this proxy object.
   */
  private void reference() throws IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    establishConnection();

    out.writeByte(Protocol.REFERENCE);
    out.writeInt(id);
    out.flush();

    if (in.readByte() != Protocol.OK) {
      throw new IOException("couldn't get reference with id " + id);
    }

    VMAddress classsrc = null;
    try {
      classsrc = (VMAddress)in.readObject();
    }
    catch (ClassNotFoundException e) {
      // not reached
      orig = MetaVM.remoteTransparency(false);
      if (debug)  e.printStackTrace();
      MetaVM.remoteTransparency(orig);
    }

    String classname = in.readUTF();
    try {
      if (MetaVM.load_class_locally || classsrc.isLocalAddress()) {
	// load the class from local disk
	clazz = Class.forName(classname);
      }
      else {
	// load via network
	ClassLoader cl = RemoteClassLoader.get(classsrc);
	in.classLoader(cl);	// set class loader

	clazz = cl.loadClass(classname);
      }
    }
    catch (ClassNotFoundException e) {
      throw new IOException(e.getMessage());
    }

    this.length = in.readInt();

    MetaVM.remoteTransparency(orig);
  }


  /**
   * Make a connection to remote JVM.
   */
  private void establishConnection() throws IOException {
    sock = new Socket(vmaddr.inetAddress(), vmaddr.port());
		// throws IOException

    if (MetaVM.tcp_nodelay)
      sock.setTcpNoDelay(true);

    OutputStream o = sock.getOutputStream();
    InputStream i = sock.getInputStream();

    int bufsize = MetaVM.bufsize;
    if (bufsize > 0) {
      bufsize *= 1024;
      o = new BufferedOutputStream(o, bufsize);
      i = new BufferedInputStream(i, bufsize);
    }

    out = new DistObjectOutputStream(o);
    in = new DistObjectInputStream(i);
  }

  private void closeConnection() throws IOException {
    out.writeByte(Protocol.CLOSE);
    out.flush();

    in.readByte();	// CLOSE

    sock.close();
    sock = null;
  }

  protected void finalize() throws Throwable {
    boolean orig = MetaVM.remoteTransparency(false);

    super.finalize();
    closeConnection();

    MetaVM.remoteTransparency(orig);
  }


  /**
   * Examines whether this proxy object is on local JVM.
   */
  public boolean isLocal() {
    boolean orig = MetaVM.remoteTransparency(false);

    boolean ret = this.vmaddr.isLocalAddress();
    if (debug)
      System.out.println("Proxy.isLocal(): " + ret);

    MetaVM.remoteTransparency(orig);

    return ret;
  }

  /**
   * Returns a local object corresponding to this proxy object
   * if this is on local JVM, null if isn't local.
   */
  public Object localObject() {
    boolean orig = MetaVM.remoteTransparency(false);

    Object ret = null;
    if (this.isLocal())
      ret = ObjectID.objectById(this.id);
    else
      ret = null;

    MetaVM.remoteTransparency(orig);

    if (debug)
      System.out.println("Proxy#localObject(): " + ret);

    return ret;
  }


  public int get32field(int slot) throws IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (debug)
      System.out.println("get32field slot: " + slot);

    if (sock == null)  reference();

    out.writeByte(Protocol.GET32FIELD);
    out.writeInt(slot);
    out.flush();

    int ret = in.readInt();
    if (debug)
      System.out.println("get32field val: " + ret);

    MetaVM.remoteTransparency(orig);

    return ret;
  }

  public long get64field(int slot) throws IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.GET64FIELD);
    out.writeInt(slot);
    out.flush();

    long ret = in.readLong();
    if (debug)
      System.out.println("get64field: " + ret +
		", 0x" + Long.toHexString(ret) +
		", " + Double.longBitsToDouble(ret));

    MetaVM.remoteTransparency(orig);

    return ret;
  }

  public Object getobjfield(int slot) throws IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (debug)
      System.out.println("get32field slot: " + slot);

    if (sock == null)  reference();

    out.writeByte(Protocol.GETOBJFIELD);
    out.writeInt(slot);
    out.flush();

    Object ret = null;
    try { ret = in.readDistObject(); }
    catch (ClassNotFoundException e) { throw new IOException(e.getMessage()); }

    MetaVM.remoteTransparency(orig);

    return ret;
  }


  public void put32field(int slot, int val)
	throws IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.PUT32FIELD);
    out.writeInt(slot);
    out.writeInt(val);
    out.flush();

    if (in.readByte() != Protocol.OK) {
      throw new IOException("put32field: couldn't get ack.");
    }

    MetaVM.remoteTransparency(orig);
  }

  public void put64field(int slot, long val)
	throws IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.PUT64FIELD);
    out.writeInt(slot);
    out.writeLong(val);
    out.flush();

    if (in.readByte() != Protocol.OK) {
      throw new IOException("put64field: couldn't get ack.");
    }

    MetaVM.remoteTransparency(orig);
  }

  public void putobjfield(int slot, Object val)
	throws IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.PUTOBJFIELD);
    out.writeInt(slot);
    out.writeDistObject(val);
    out.flush();

    if (in.readByte() != Protocol.OK) {
      throw new IOException("putobjfield: couldn't get ack.");
    }

    MetaVM.remoteTransparency(orig);
  }


  public int arraylength() throws IOException {
    return this.length;

/*
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.ARYLENGTH);
    out.flush();

    int ret = in.readInt();

    MetaVM.remoteTransparency(orig);

    return ret;
*/
  }


  public int aload32(int index)
	throws ArrayIndexOutOfBoundsException, IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.ARYLOAD32);
    out.writeInt(index);
    out.flush();

    Object exc = null;
    try { exc = in.readDistObject(); }
    catch (ClassNotFoundException e) { throw new IOException(e.getMessage()); }
    if (exc != null)	// Exception is occurred
      throw (ArrayIndexOutOfBoundsException)exc;

    int ret = in.readInt();

    MetaVM.remoteTransparency(orig);

    return ret;
  }

  public long aload64(int index)
	throws ArrayIndexOutOfBoundsException, IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.ARYLOAD64);
    out.writeInt(index);
    out.flush();

    Object exc = null;
    try { exc = in.readDistObject(); }
    catch (ClassNotFoundException e) { throw new IOException(e.getMessage()); }
    if (exc != null)	// Exception is occurred
      throw (ArrayIndexOutOfBoundsException)exc;

    long ret = in.readLong();

    MetaVM.remoteTransparency(orig);

    return ret;
  }

  public Object aloadobj(int index)
	throws ArrayIndexOutOfBoundsException, IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.ARYLOADOBJ);
    out.writeInt(index);
    out.flush();

    Object exc = null;
    try { exc = in.readDistObject(); }
    catch (ClassNotFoundException e) { throw new IOException(e.getMessage()); }
    if (exc != null)	// Exception is occurred
      throw (ArrayIndexOutOfBoundsException)exc;

    Object ret = null;
    try { ret = in.readDistObject(); }
    catch (ClassNotFoundException e) { throw new IOException(e.getMessage()); }

    MetaVM.remoteTransparency(orig);

    return ret;
  }

  public void astore32(int index, int val)
	throws ArrayIndexOutOfBoundsException, IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.ARYSTORE32);
    out.writeInt(index);
    out.writeInt(val);
    out.flush();

    Object exc = null;
    try { exc = in.readDistObject(); }
    catch (ClassNotFoundException e) { throw new IOException(e.getMessage()); }
    if (exc != null)	// Exception is occurred
      throw (ArrayIndexOutOfBoundsException)exc;

    MetaVM.remoteTransparency(orig);
  }

  public void astore64(int index, long val)
	throws ArrayIndexOutOfBoundsException, IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.ARYSTORE64);
    out.writeInt(index);
    out.writeLong(val);
    out.flush();

    Object exc = null;
    try { exc = in.readDistObject(); }
    catch (ClassNotFoundException e) { throw new IOException(e.getMessage()); }
    if (exc != null)	// Exception is occurred
      throw (ArrayIndexOutOfBoundsException)exc;

    MetaVM.remoteTransparency(orig);
  }

  public void astoreobj(int index, Object val)
	throws ArrayIndexOutOfBoundsException, IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.ARYSTOREOBJ);
    out.writeInt(index);
    out.writeDistObject(val);
    out.flush();

    Object exc = null;
    try { exc = in.readDistObject(); }
    catch (ClassNotFoundException e) { throw new IOException(e.getMessage()); }
    if (exc != null)	// Exception is occurred
      throw (ArrayIndexOutOfBoundsException)exc;

    MetaVM.remoteTransparency(orig);
  }


  public Object invoke(int slot, String sig, Object[] args)
	throws Throwable {
    boolean orig = MetaVM.remoteTransparency(false);

    if (debug) {
      System.out.println("Proxy.invoke() called.");
      System.out.println(" clazz: " + clazz.getName());
      System.out.println(" slot: " + slot);
      System.out.println(" sig : " + sig);
      System.out.println(" args: " + args);
      System.out.flush();

      if (args != null) {
	for (int i = 0; i < args.length; i++) {
	  System.out.println("  args[" + i + "]: " +
			args[i] + " " + args[i].getClass());
	}
	System.out.flush();
      }

      if (args != null) {
	//Class clazz = Object[].class;
	Class clazz = args.getClass();
	System.out.println(clazz);

	java.io.ObjectStreamClass oclz =
			java.io.ObjectStreamClass.lookup(clazz);
	System.out.println(oclz);
      }
    }

    if (sock == null)  reference();

    out.writeByte(Protocol.INVOKE);
    out.writeShort(slot);
    if (slot == 0) {	// caller is a constructor
      out.writeUTF(sig);
    }
    out.writeDistObject(new ArrayOfObjectWrapper(args));
    out.flush();

    if (in.readByte() != Protocol.OK) {
      Object thrw = null;
      try { thrw = in.readDistObject(); }
      catch (ClassNotFoundException e) {
	throw new IOException(e.getMessage());
      }
      if (thrw != null) {	// Throwable is thrown at remote machine
	throw (Throwable)thrw;
      }
    }

    Object ret = null;
    try { ret = in.readDistObject(); }
    catch (ClassNotFoundException e) { throw new IOException(e.getMessage()); }

    MetaVM.remoteTransparency(orig);

    if (debug) {
      System.out.println("Proxy.invoke() done.");
      System.out.flush();
    }
    return ret;
  }


  public void monitorEnter() throws IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.MONITORENTER);
    out.flush();

    if (in.readByte() != Protocol.OK) {
      throw new IOException("couldn't get a monitor.");
    }

    MetaVM.remoteTransparency(orig);
  }


  public void monitorExit() throws IOException {
    boolean orig = MetaVM.remoteTransparency(false);

    if (sock == null)  reference();

    out.writeByte(Protocol.MONITOREXIT);
    out.flush();

    if (in.readByte() != Protocol.OK) {
      throw new IOException("couldn't release a monitor.");
    }

    MetaVM.remoteTransparency(orig);
  }
}
